/*

   Unix socket server for nightview.

   Create socket, listen its and calls commands for operation

   This code is strongly inspired by N. Matthew's and R. Stone's
   Beginning Linux Programming 2nd Edition, Wrong Press Ltd. (1999)
   and Infos to glibc.

   $Id: server-local.c,v 1.9 2007-06-21 17:23:01 hroch Exp $

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <signal.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include "nightview.h"
#include "daemon.h"

#define LEN 80

/* length incoming queue */
#define MAX_CLIENTS 5


void clean_sock(int sig)
{
  syslog(LOG_NOTICE,"Exiting.\n");  
  if( unlink(NIGHT_SHOCK) < 0 )
    syslog(LOG_ERR,"Unlink:socket failed\n");
  signal(sig,SIG_DFL);
}

void terminate_daemon(int sig)
{
  driver_close();
  clean_sock(sig);
  exit(0);
}

void abort_daemon(int sig)
{
  driver_close();
  clean_sock(sig);
  syslog(LOG_ERR,"Daemon aborted due to software error.\n");
  exit(1);
}

void hup_sock(int sig)
{
  signal(sig,SIG_DFL);
}


int main(int argc, char *argv[])
{

  struct stat name_stat;
  struct sockaddr_un name, client;
  int sock, cid;
  char data[LEN], *ansfer = NULL;
  size_t size;
#ifndef DEBUG
  int pid;
#endif

  /* initialize signal handlers */
  signal(SIGTERM,terminate_daemon);
  signal(SIGINT,terminate_daemon);
  signal(SIGQUIT,terminate_daemon);
  signal(SIGABRT,abort_daemon);
  signal(SIGSEGV,abort_daemon);
  signal(SIGHUP, hup_sock);

  /* initialize log */
  openlog("nightviewd",LOG_PID, LOG_NOTICE);
  syslog(LOG_NOTICE,"ver. %s. Local server starting.\n",NIGHTVERSION);

#ifndef DEBUG
  /* close unused files, detach from terminal and fork */
  fclose(stdin);
  fclose(stdout);
  fclose(stderr);
  
  if( (pid = fork()) == 0 )
    goto go;
  else if( pid == -1 ) {
    syslog(LOG_ERR,"Daemonization failed: %m\n");
    return(1);
  }
  else
    return(0);

 go:
#endif

  /* initialize commands */
  driver_init(argc, argv);

  /* check if server not running */
  if( stat(NIGHT_SHOCK,&name_stat) == 0 ) {
    syslog(LOG_ERR,"Socket descriptor %s exist.\n Remove it manually if server don't runnig and restart me.",NIGHT_SHOCK);
    return(1);
  }

  /* umask for created socket */
  umask(0111);

  /* create server socket */
  if( (sock = socket(PF_LOCAL, SOCK_STREAM, 0)) < 0 ) {
    syslog(LOG_ERR,"socket: %m\n");
    return(1);
  }
  
  /* init server socket */
  name.sun_family = AF_LOCAL;
  strncpy(name.sun_path,NIGHT_SHOCK, sizeof(name.sun_path));
  size = sizeof (name);
  if( bind(sock, (struct sockaddr *) &name, size) < 0) {
    syslog(LOG_ERR,"bind: %m\n");
    return(1);
  }

  /* init incoming call queue */
  if( listen(sock, MAX_CLIENTS) < 0 ) {
    syslog(LOG_ERR,"listen: %m\n");
    return(1);
  }
    
  /* we are waiting to clients */
  while(1) {

    /*  accepting incoming call */
    socklen_t socklen = sizeof(client);
    if( (cid = accept(sock, (struct sockaddr *) &client, &socklen)) < 0 ) {
      syslog(LOG_ERR,"accept: %m\n");
      return(1);
    }

    /* data read */
    if( (size = read(cid,data,LEN)) < 0 ) {
      syslog(LOG_ERR,"read: %m\n");
      return(1);
    }

    if( size > 0 ) {

      data[size] = '\0';
      /* transfer data to the control routine */
      if( (ansfer = driver(data,&size)) == NULL || size < 0 ) {
	syslog(LOG_ERR,"commands: %m\n");
	return(1);
      }

      /* response */
      if( write(cid,ansfer,size) < 0 ) {
	syslog(LOG_ERR,"write: %m\n");
	return(1);
      }
    }

    /* close socket to the client */
    if( close(cid) < 0 ) {
      syslog(LOG_ERR,"close: %m\n");
      return(1);
    }
    if( ansfer ) 
      free(ansfer);

  } /* end of loop */

  return(0);
}
