/*

  FITS file open/save

  $Id: fits.c,v 1.14 2002-07-16 15:08:29 hroch Exp $

*/

#include "fitsadd.h"
#include "gtknightview.h"
#include "string.h"
#include <fitsio.h>
#include <sys/stat.h>

#define FWHM_BOX 5
#define NSKIP 10

/* fits buffer parameters */
static double mean, dev;

/* handling procedures */

/*---------------------------------------------------------------------*/

FIMAGE *read_fits(name)
     char *name;
{
  fitsfile *file;
  int status, naxis, npixels, fpixel, nullval, anynull;
  long naxes[2];
  float datamin, datamax;
  FIMAGE *f;

  status = 0;
  fits_open_file(&file,name,READONLY,&status);
  fits_get_img_dim(file,&naxis,&status);
  fits_get_img_size(file,2,naxes,&status);

  f = NULL;
  if( status == 0 ) {
    npixels  = naxes[0] * naxes[1];         /* number of pixels in the image */
    fpixel   = 1;
    nullval  = 0;                /* don't check for null values in the image */
    datamin  = 1.0e30;
    datamax  = -1.0e30;
    f = malloc(sizeof(FIMAGE));
    f->image = malloc(npixels*sizeof(float));
    f->width = naxes[0];
    f->height = naxes[1];
    f->filename = strdup(name);
  }
  if( f && f->image )
    fits_read_img(file, TFLOAT, 
		  fpixel, npixels, &nullval, f->image, &anynull, &status);

  fits_close_file(file, &status);
  fits_report_error(stderr, status);  /* print out any error message */
#ifdef DEBUG
  printf("status=%d\n",status);
#endif
  if( status == 0 ) {
    return(f); }
  else {
    if( f && f->image ) 
      free(f->image);
    if( f ) 
      free(f);
    return(NULL); }
}

/*--------------------------------------------------------------------*/

static int compare_float(const void *a, const void *b)
{
  const float *da = (const float *) a;
  const float *db = (const float *) b;

  return (*da > *db) - (*da < *db);
}


unsigned char *float2RGB(FIMAGE *f)
{
  unsigned char *data;
  float *d;
  float *image;
  int nx,ny,i,j,l,value,ndata;

  if( f == NULL ) return(NULL);

  nx = f->width;
  ny = f->height;
  image = f->image;

  ndata = nx*ny/(NSKIP*NSKIP);
  d = malloc(ndata*sizeof(float));
  for( i = 0, j = 0; j < ndata; i = i + NSKIP, j++)
    d[j] = image[i];
  qsort(d,ndata,sizeof(float),compare_float);
  mean = (*(d + ndata/2) + *(d + ndata/2 - 1))/2.0;
 
  for( i = 0, j = 0; j < ndata; i = i + NSKIP, j++)
    *(d + j) = abs(*(image + i) - mean);
  qsort(d,ndata,sizeof(float),compare_float);
  dev = (*(d + ndata/2) + *(d + ndata/2 - 1))/2.0;
  free(d);
  if( abs(dev) < 1e-2 ) dev = 1.0;

#ifdef DEBUG
  printf("mean=%f sig=%f  n=%d\n", mean, dev, ndata);
#endif

  mean = mean - 3*dev;
  dev = dev/15.0;
    
  data = malloc(3*nx*ny);
  l = 0;
  /* verticaly flip the data and define RGB values */
  for( i = 0; i < ny; i++ ) 
    for( j = 0; j < nx; j++ ) {
      value = rint((image[nx*(ny - i - 1) + j] - mean)/dev);
      if( value > 255 ) value = 255;
      if( value < 0 ) value = 0;
      data[l] = data[l+1] = data[l+2] = value;
      l = l + 3;
    }
  return(data);
}

/*-------------------------------------------------------------------*/

int write_fits(FIMAGE *f,char *name)
{

  /* the files are created read-only */
  umask( S_IWUSR | S_IWGRP | S_IWOTH );

  /*  copy the output (temporary) file to output. */  
  fits_cp(f->filename,name);

  return(TRUE);
}

/*----------------------------------------------------------------------*/

float get_intensity(FIMAGE *f, int i,int j)
{
  if( f && ( 0 <= j && j < f->height && 0 <= i && i < f->width ) )
    return(f->image[f->width*(f->height - j - 1) + i]);
    /*    return(*(image + nx*j + i));*/
  else
    return(0.0);
}

/*--------------------------------------------------------------------*/

void get_fwhm(FIMAGE *f, int i,int j,float *xfwhm,float *yfwhm)
{
  int ii,jj;
  float sum, pom;

  if( f ) {
    sum = 0.0; *xfwhm = 0.0; 
    for( ii = -FWHM_BOX; ii <= FWHM_BOX; ii++ ) {
      pom = get_intensity(f,i+ii,j) - mean;
      if( pom < 0.0) pom = 0.0;
      sum = sum + pom;
      *xfwhm = *xfwhm + pom*ii*ii;
    }
    *xfwhm = sqrt((double) *xfwhm / (sum - 1));
    sum = 0.0; *yfwhm = 0.0;
    for( jj = -FWHM_BOX; jj <= FWHM_BOX; jj++ ) {
      pom = get_intensity(f,i,j+jj) - mean;
      if( pom < 0.0) pom = 0.0;
      sum = sum + pom;
      *yfwhm = *yfwhm + pom*jj*jj;
    }
    *yfwhm = sqrt((double) *yfwhm / (sum - 1));
    return;
  }
  else {
    *xfwhm = 0.0;
    *yfwhm = 0.0;
    return;
  }
}

int get_points(FIMAGE *f,int ii, int jj, int d, int x, int y, int range, 
	       GdkPoint *p)
{
  int i, nonzero;
  float i1,max;
  
  /* transform pixels near ii, jj to range 0..255, mean background
     is 20 (approx. 10% of max. range), max. peak has 255,
     the x coordinate of GdkPoint is relative x coordinate of image,
     the y is transformed intensity  */

  /* get max peak */
  if( f ) {
    max = 0;
    for( i = 0; i < 2*d; i++) {
      if( (i1 = get_intensity(f,ii + x*(i - d),jj + y*(i - d))) > max)
	max = i1;
    }

    if( (nonzero = abs(max) > 1e-5) ) {
      max = 0.9*range/max;
      for( i = 0; i < 2*d; i++) {
	p[i].x = i - d;
	p[i].y = max*(get_intensity(f,ii + x*(i - d),jj + y*(i - d)) - mean);
      } 
    }
  }

  if( f == NULL || !nonzero ) {
    for( i = 0; i < 2*d; i++) {
      p[i].x = i - d;
      p[i].y = 0.1*max;
    }
  }

  return(i);
}
