
#include <unistd.h>
#include <syslog.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "ccdtypes.h"
#include "ccdcommon.h"

/* 
   Methods for manipualtig and accessing of structure members.

   High level camera operation functions create, initialize get
   and set the ccd variable(s).

   It's reccomended to use this functions. NOT direct an access to 
   a structure members.

*/

/*=======================================================================
  PART 0

  definitions 
*/

#define NOTEMP -999.9
#define NODEG 999.9


/* ========================================================================
   PART I
 
   create, destroy objects 
*/

/*
   create a new instance of CCD, needs:

   the id as a hw address of device (port for example)
   the address as a general description of an address (may be NULL)

*/


static const int ccdtypes_maxccd = MAXCCD;
static const int ccdtypes_maxfilters = MAXFILTERS;
static const int ccdtypes_maxreadout = MAXREADOUT;
static const int ccdtypes_maxbad = MAXBAD;


CCD *ccd_new(char *address)
{
  CCD *ccd;
  int i;

  if( ! address )
    return(NULL);

  if( (ccd = malloc(sizeof(CCD))) && 
      (ccd->address = malloc(strlen(address) + 1)) ) {
    strcpy(ccd->address,address);
    ccd->on = 0;
    for( i = 0; i < ccdtypes_maxccd; i++ )
      ccd->ccdinfo[i] = NULL;
    ccd->tempccd = NOTEMP;
    ccd->tempair = NOTEMP;
    ccd->reg_power = NOTEMP;
    ccd->fan_status = 0;
    ccd->nfilters = 0;
    ccd->ifilter = -1;
    for( i = 0; i < ccdtypes_maxfilters; i++ )
      ccd->filter[i] = NULL;
    ccd->maxfilters = ccdtypes_maxfilters;
    ccd->maxccd = ccdtypes_maxccd;
    ccd->fitsname = NULL;
    ccd->datetime = NULL;
    ccd->object = NULL;
    ccd->site = NULL;
    ccd->telescope = NULL;
    return(ccd);
  }
  return(NULL);
}

/* destroy object (free a memory) associated with CCD */

void ccd_free(CCD *ccd)
{
  int i;

  if( ccd ) {
    for( i = 0; i < MAXCCD; i++)
      ccd_free_info(ccd,i);
    for( i = 0; i < ccd->nfilters; i++)
      free(ccd->filter[i]);
    free(ccd->address);
    free(ccd->fitsname);
    free(ccd->datetime);
    free(ccd->object);
    free(ccd->site);
    free(ccd->telescope);
    free(ccd);
    ccd = NULL; /* no fluence */
  }
}


/* =====================================================================
   PART II

   manipulate with objects
*/

/* for CCD */

char *ccd_get_address(CCD *ccd)
{
  if( ccd )
    return(ccd->address);
  else
    return(NULL);
}

int ccd_set_on(CCD *ccd, int on)
{
  if( ccd ) {
    ccd->on = on;
    return(1);
  }
  else
    return(0);
}
      
int ccd_get_on(CCD *ccd)
{
  if( ccd )
    return(ccd->on);
  else
    return(0);
}

/* for CCDINFO */

CCDINFO *ccd_new_info(CCD *ccd, int n, char *name, char *firmware, 
		      char *serial, int width, int height, int offset, 
		      int adrange, int abg, float pixwidth, float pixheight, 
		      float gain, int nreadouts, double minexp, double maxexp,
		      int nbad, int bad[])
{
  CCDINFO *ccdinfo;
  int i;

  if( ccd->ccdinfo[n] )
    ccd_free_info(ccd,n);

  ccdinfo = NULL;
  if( ccd && (0 <= n && n < ccd->maxccd ) 
      && (ccdinfo = malloc(sizeof(CCDINFO))) ) {

    if( name && (ccdinfo->camera_name = malloc(strlen(name)+1)) )
      strcpy(ccdinfo->camera_name,name);
    else
      ccdinfo->camera_name = NULL;

    if( firmware && (ccdinfo->firmware = malloc(strlen(firmware)+1)) )
      strcpy(ccdinfo->firmware,firmware);
    else
      ccdinfo->firmware = NULL;

    if( serial && (ccdinfo->serial = malloc(strlen(serial)+1)) )
      strcpy(ccdinfo->serial,serial);
    else
      ccdinfo->serial = NULL;

    ccdinfo->width=width;
    ccdinfo->height=height;
    ccdinfo->pixel_width=pixwidth;
    ccdinfo->pixel_height=pixheight;
    ccdinfo->gain = gain;
    ccdinfo->offset = offset;
    ccdinfo->adrange = adrange;
    ccdinfo->abg_type = abg;
    ccdinfo->minexp = minexp;
    ccdinfo->maxexp = maxexp;
    ccdinfo->maxbad = ccdtypes_maxbad;
    ccdinfo->maxreadout = ccdtypes_maxreadout;
    if( nreadouts <= ccdinfo->maxreadout )
      ccdinfo->n_readout = nreadouts;
    else
      ccdinfo->n_readout = ccdinfo->maxreadout;
    for( i = 0; i < ccdinfo->maxreadout; i++)
      ccdinfo->ccdreadout[i] = NULL;
    ccdinfo->n_bad_col = nbad;
    for( i = 0; i < ccdinfo->maxbad; i++)
      ccdinfo->bad_col[i] = 0;
    for( i = 0; i < nbad && i < ccdinfo->maxbad; i++)
      ccdinfo->bad_col[i] = bad[i];

    ccd->ccdinfo[n] = ccdinfo;
  }
  return(ccdinfo);
}

CCDREADOUT *ccd_new_readout(CCDINFO *ccdinfo, int i, int mode, int width, 
			    int height, float gain, float pixw, float pixh)
{
  CCDREADOUT *ccdread;

  if( ccdinfo->ccdreadout[i] ) {
    free(ccdinfo->ccdreadout[i]);
    ccdinfo->ccdreadout[i]= NULL;
  }

  ccdread = NULL;
  if( ccdinfo && i < ccdinfo->maxreadout && 
      (ccdread = malloc(sizeof(CCDREADOUT))) ) {
    
    ccdread->mode = mode;
    ccdread->width = width;
    ccdread->height = height;
    ccdread->gain = gain;
    ccdread->pixwidth = pixw;
    ccdread->pixheight = pixh;

    ccdinfo->ccdreadout[i] = ccdread;
  }
  return(ccdread);
}


void ccd_free_info(CCD *ccd, int i)
{
  int j;

  if( ccd && ccd->ccdinfo[i] ) {
    for( j = 0; j < MAXREADOUT; j++ )
      ccd_free_readout(ccd,i,j);
    free(ccd->ccdinfo[i]->camera_name);
    free(ccd->ccdinfo[i]->firmware);
    free(ccd->ccdinfo[i]->serial);
    free(ccd->ccdinfo[i]);
    ccd->ccdinfo[i] = NULL;
  }
}

void ccd_free_readout(CCD *ccd, int i, int j)
{
  if( ccd && ccd->ccdinfo[i] && ccd->ccdinfo[i]->ccdreadout[j] )
    free(ccd->ccdinfo[i]->ccdreadout[j]);
  ccd->ccdinfo[i]->ccdreadout[j] = NULL;
}


int ccdinfo_set_name(CCD *ccd, int i,char *name)
{
  if( name && (0 <= i && i < ccd->maxccd) && ccd && ccd->ccdinfo[i] ) {
    strcpy(ccd->ccdinfo[i]->camera_name,name);
    return(1);
  }
  else
    return(0);
}

int ccdinfo_set_size(CCD *ccd, int i, int width, int height)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] ) {
    ccd->ccdinfo[i]->width = width;
    ccd->ccdinfo[i]->height = height;
    return(1);
  }
  else
    return(0);
}

int ccdinfo_set_pixsize(CCD *ccd, int i, int width, int height)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] ) {
    ccd->ccdinfo[i]->pixel_width = width;
    ccd->ccdinfo[i]->pixel_height = height;
    return(1);
  }
  else
    return(0);
}

int ccdinfo_set_gain(CCD *ccd, int i, float gain)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] ) {
    ccd->ccdinfo[i]->gain = gain;
    return(1);
  }
  else
    return(0);
}


/* deprecated */
char *ccdinfo_get_name(CCD *ccd, int i)
{
  return(ccd_get_cameraname(ccd,i));
}


/* deprecated */
int ccdinfo_get_size_width(CCD *ccd, int i)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] ) {
    return(ccd->ccdinfo[i]->width);
  }
  else
    return(0);
}


/* deprecated */
int ccdinfo_get_size_height(CCD *ccd, int i)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] ) {
    return(ccd->ccdinfo[i]->height);
  }
  else
    return(0);
}

/* deprecated */
float ccdinfo_get_size_pixwidth(CCD *ccd, int i)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] ) {
    return(ccd->ccdinfo[i]->pixel_width);
  }
  else
    return(0.0);
}

/* deprecated */
float ccdinfo_get_size_pixheight(CCD *ccd, int i)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] ) {
    return(ccd->ccdinfo[i]->pixel_height);
  }
  else
    return(0.0);
}

/* deprecated */
float ccdinfo_get_gain(CCD *ccd, int i)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] ) {
    return(ccd->ccdinfo[i]->gain);
  }
  else
    return(0.0);
}

/* deprecated */
double ccdinfo_get_minexp(CCD *ccd, int i)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] ) {
    return(ccd->ccdinfo[i]->minexp);
  }
  else
    return(0.0);
}

double ccd_get_minexp(CCD *ccd, int i)
{
  return(ccdinfo_get_minexp(ccd,i));
}

/* deprecated */
double ccdinfo_get_maxexp(CCD *ccd, int i)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] ) {
    return(ccd->ccdinfo[i]->maxexp);
  }
  else
    return(0.0);
}

double ccd_get_maxexp(CCD *ccd, int i)
{
  return(ccdinfo_get_maxexp(ccd,i));
}


int ccd_nchiplist(CCD *ccd)
{
  int i;
  if( ccd ) {

    for( i = 0; i < ccd->maxccd && ccd->ccdinfo[i]; i++)
      ;
    return(i);
  }
  else
    return(0);
}

char *ccd_get_firmware(CCD *ccd, int i)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] )
    return(ccd->ccdinfo[i]->firmware);
  else
    return("");
}

char *ccd_get_cameraname(CCD *ccd, int i)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] && 
      ccd->ccdinfo[i]->camera_name )
    return(ccd->ccdinfo[i]->camera_name);
  else
    return("");
}

char *ccd_get_serial(CCD *ccd, int i)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] && 
      ccd->ccdinfo[i]->serial )
    return(ccd->ccdinfo[i]->serial);
  else
    return("");
}

int ccd_get_abgtype(CCD *ccd, int i)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] )
    return(ccd->ccdinfo[i]->abg_type);
  else
    return(0);
}

int ccd_get_bitpix(CCD *ccd, int i)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] )
    return(ccd->ccdinfo[i]->bitpix);
  else
    return(0);  
}

int ccd_get_adrange(CCD *ccd, int i)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] )
    return(ccd->ccdinfo[i]->adrange);
  else
    return(0);  
}

int ccd_get_bias(CCD *ccd, int i)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] )
    return(ccd->ccdinfo[i]->offset);
  else
    return(0);  
}

int ccd_get_nbadcol(CCD *ccd, int i)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] )
    return(ccd->ccdinfo[i]->n_bad_col);
  else
    return(0);
}

int ccd_get_badcol(CCD *ccd, int i, int j)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] && 
      (0 <= j && j < ccd->ccdinfo[i]->maxbad) )
    return(ccd->ccdinfo[i]->bad_col[j]);
  else
    return(0);
}

int ccd_get_nreadout(CCD *ccd, int i)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] )
    return(ccd->ccdinfo[i]->n_readout);
  else
    return(0);
}

int ccd_get_readout_mode(CCD *ccd, int i, int j)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] && 
      (0 <= j && j < ccd->ccdinfo[i]->maxreadout) && 
      ccd->ccdinfo[i]->ccdreadout[j] )
    return(ccd->ccdinfo[i]->ccdreadout[j]->mode);
  else
    return(0);
}
  
int ccd_get_readout_width(CCD *ccd, int i, int j)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] &&  
      (0 <= j && j < ccd->ccdinfo[i]->maxreadout) && 
      ccd->ccdinfo[i]->ccdreadout[j] )
    return(ccd->ccdinfo[i]->ccdreadout[j]->width);
  else
    return(0);
}

int ccd_get_readout_height(CCD *ccd, int i, int j)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] && 
      (0 <= j && j < ccd->ccdinfo[i]->maxreadout) && 
      ccd->ccdinfo[i]->ccdreadout[j] )
    return(ccd->ccdinfo[i]->ccdreadout[j]->height);
  else
    return(0);
}


float ccd_get_readout_pixsize(CCD *ccd, int i, int j)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] && 
      (0 <= j && j < ccd->ccdinfo[i]->maxreadout) && 
      ccd->ccdinfo[i]->ccdreadout[j] )
    return(ccd->ccdinfo[i]->ccdreadout[j]->pixwidth);
  else
    return(0.0);
}


float ccd_get_readout_piysize(CCD *ccd, int i, int j)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] && 
      (0 <= j && j < ccd->ccdinfo[i]->maxreadout) && 
      ccd->ccdinfo[i]->ccdreadout[j] )
    return(ccd->ccdinfo[i]->ccdreadout[j]->pixheight);
  else
    return(0.0);
}

float ccd_get_readout_gain(CCD *ccd, int i, int j)
{
  if( ccd && (0 <= i && i < ccd->maxccd) && ccd->ccdinfo[i] && 
      (0 <= j && j < ccd->ccdinfo[i]->maxreadout) && 
      ccd->ccdinfo[i]->ccdreadout[j] )
    return(ccd->ccdinfo[i]->ccdreadout[j]->gain);
  else
    return(0.0);
}

int ccd_set_tempccd(CCD *ccd, float temp)
{
  if( ccd ) {
    ccd->tempccd = temp;
    return(1);
  }
  else
    return(0);
}

int ccd_set_tempair(CCD *ccd, float temp)
{
  if( ccd ) {
    ccd->tempair = temp;
    return(1);
  }
  else
    return(0);
}

int ccd_set_fan(CCD *ccd, int fan)
{
  if( ccd ) {
    ccd->fan_status = fan;
    return(1);
  }
  else
    return(0);
}

float ccd_get_tempccd(CCD *ccd)
{
  if( ccd )
    return(ccd->tempccd);
  else
    return(NOTEMP);
}

float ccd_get_tempair(CCD *ccd)
{
  if( ccd )
    return(ccd->tempair);
  else
    return(NOTEMP);
}

float ccd_get_regpower(CCD *ccd)
{
  if( ccd )
    return(ccd->reg_power);
  else
    return(0.0);
}

int ccd_get_fan(CCD *ccd)
{
  if( ccd )
    return(ccd->fan_status);
  else
    return(-1);
}

int ccd_get_nfilters(CCD *ccd)
{
  if( ccd )
    return(ccd->nfilters);
  else
    return(-1);
}

char *ccd_get_cfilter(CCD *ccd)
{
  char *f;
  if( ccd && (0 <= ccd->ifilter && ccd->ifilter < ccd->nfilters) &&
      (f = ccd->filter[ccd->ifilter]) )
    return(f);
  else
    return("");
}

int ccd_set_filter(CCD *ccd, int i)
{
  if( ccd && (0 <= i && i < ccd->nfilters) && ccd->filter[i] ) {
    ccd->ifilter = i;
    return(1);
  }
  else
    return(0);
}

char *ccd_get_filter(CCD *ccd, int i)
{
  char *f;
  if( ccd && (0 <= i && i < ccd->nfilters) && (f = ccd->filter[i]) )
    return(f);
  else
    return("");
}

int ccd_set_cfilter(CCD *ccd, char *f)
{
  int i;

  if( f && ccd )
    for( i = 0; i < ccd->nfilters; i++)
      if( ccd->filter[i] && strcmp(ccd->filter[i],f) == 0 ) {
	ccd->ifilter = i;
	return(1);
      }
  return(0);
}

char *ccd_filter_list(CCD *ccd)
{
  char *line, *filter;
  int i,j,len,nfilters;

  nfilters = ccd_get_nfilters(ccd);
  len = 0;
  for(i = 0; i < nfilters; i++) {
    filter = ccd_get_filter(ccd,i);
    if( filter )
      len = len + strlen(filter) + 3;
  }

  if( len > 0 && (line = malloc(len+1)) ) {
    j = 0;
    for( i = 0; i < nfilters; i++) {
      filter = ccd_get_filter(ccd,i);
      if( filter ) {
	snprintf(line+j,len-j,"'%s' ",filter);
	j = j + strlen(filter) + 3;
      }
    }
    return(line);
  }
  else
    return(NULL);
}

int ccd_set_filters(CCD *ccd, int wheel, int n, char *fl[])
{
  /* n is dimension of fl[] */

  int i,k;

  /*
  for(i = 0; i < n; i++ )
    if( ! fl[i] )
      return(0);
  */

  if( ccd ) {

    ccd->wheel = wheel;

    /* free older filter set */
    for(i = 0; i < ccdtypes_maxfilters; i++) {
      free(ccd->filter[i]);
      ccd->filter[i] = NULL;
    }
    ccd->nfilters = 0;
    
    k = 0;
    for( i = 0; i < n  && k < ccdtypes_maxfilters; i++ ) {
      
      if( fl[i] ) {

	ccd->filter[k] = strdup(fl[i]);
	k++;
	ccd->nfilters = k;

      }
    }  
    return(1);

  }
  else
    return(0);


  /* -- */

    /*
  if( n > ccdtypes_maxfilters ) 
    n = ccdtypes_maxfilters;
  if( ccd ) {

    for(i = 0; i < ccdtypes_maxfilters; i++) {
      free(ccd->filter[i]);
      ccd->filter[i] = NULL;
    }

    ccd->nfilters = n;
    for( i = 0; i < n; i++ ) {
      if( fl[i] )
	ccd->filter[i] = strdup(fl[i]);
      else
	ccd->filter[i] = NULL;
    }
    return(1);
  }
  else
    return(0);
    */
}

int ccd_set_filterwheel(CCD *ccd, int wheel)
{
  if( ccd ) {
    ccd->wheel = wheel;
    return(1);
  }
  else
    return(0);
}

int ccd_get_filterwheel(CCD *ccd)
{
  if( ccd )
    return(ccd->wheel);
  else
    return(-1);
}

int ccd_set_site(CCD *ccd, char *name, double lg, double ln, double al)
{
  if( ccd && name ) {
    free(ccd->site);
    ccd->site = strdup(name);
    ccd->longitude = lg;
    ccd->latitude = ln;
    ccd->altitude = al;
    return(1);
  }
  else
    return(0);
}

char *ccd_get_site_name(CCD *ccd)
{
  if( ccd && ccd->site )
    return(ccd->site);
  else
    return("");
}

double ccd_get_site_long(CCD *ccd)
{
  if( ccd )
    return(ccd->longitude);

  return(NODEG);
}

double ccd_get_site_lat(CCD *ccd)
{
  if( ccd )
    return(ccd->latitude);

  return(NODEG);
}

double ccd_get_site_alt(CCD *ccd)
{
  if( ccd )
    return(ccd->altitude);

  return(NODEG);
}

int ccd_set_telescope(CCD *ccd, char *name, double focus)
{
  if( ccd && name ) {
    free(ccd->telescope);
    ccd->telescope = strdup(name);
    ccd->focus = focus;
    return(1);
  }
  else
    return(0);
}

char *ccd_get_telescope_name(CCD *ccd)
{
  if( ccd && ccd->telescope )
    return(ccd->telescope);
  else
    return("");
}

double ccd_get_telescope_focus(CCD *ccd)
{
  if( ccd )
    return(ccd->focus);
  else
    return(NODEG);
}

int ccd_set_fitsname(CCD *ccd, char *name)
{
  if( ccd && name ) {
    free(ccd->fitsname);
    ccd->fitsname = strdup(name);
    return(1);
  }
  else
    return(0);
}

char *ccd_get_fitsname(CCD *ccd)
{
  if( ccd && ccd->fitsname )
    return(ccd->fitsname);
  else
    return("");
}

int ccd_set_object(CCD *ccd, char *name)
{
  if( name && ccd ) {
    free(ccd->object);
    ccd->object = strdup(name);
    return(1);
  }
  else
    return(0);
}

char *ccd_get_object(CCD *ccd)
{
  if( ccd && ccd->object )
    return(ccd->object);
  else
    return("");
}

int ccd_set_observer(CCD *ccd, char *name)
{
  if( name && ccd ) {
    free(ccd->observer);
    ccd->observer = strdup(name);
    return(1);
  }
  else
    return(0);
}

char *ccd_get_observer(CCD *ccd)
{
  if( ccd && ccd->observer )
    return(ccd->observer);
  else
    return("");
}

int ccd_set_datetime(CCD *ccd, char *datetime)
{
  if( datetime && ccd ) {
    free(ccd->datetime);
    ccd->datetime = strdup(datetime);
    return(1);
  }
  else
    return(0);
}

char *ccd_get_datetime(CCD *ccd)
{
  if( ccd && ccd->datetime )
    return(ccd->datetime);
  else
    return("");
}

int ccd_set_exptime(CCD *ccd, double exptime)
{
  if( ccd ) {
    ccd->exptime = exptime;
    return(1);
  }
  else
    return(0);
}

double ccd_get_exptime(CCD *ccd)
{
  if( ccd )
    return(ccd->exptime);

  return(-1.0);
}


int ccd_set_ra(CCD *ccd, double ra)
{
  if( ccd ) {
    ccd->ra = ra;
    return(1);
  }
  else
    return(0);
}

double ccd_get_ra(CCD *ccd)
{
  if( ccd )
    return(ccd->ra);

  return(-999.9);
}


int ccd_set_dec(CCD *ccd, double dec)
{
  if( ccd ) {
    ccd->dec = dec;
    return(1);
  }
  else
    return(0);
}

double ccd_get_dec(CCD *ccd)
{
  if( ccd )
    return(ccd->dec);

  return(-999.9);
}

int ccd_set_binnig(CCD *ccd, int bin)
{
  if( ccd ) {
    ccd->binning = bin;
    return(1);
  }
  else
    return(0);
}

int ccd_get_binning(CCD *ccd)
{
  if( ccd )
    return(ccd->binning);

  return(0);
}

int ccd_get_saturation(CCD *ccd)
{
  if( ccd )
    return(ccd->ccdinfo[ccd->binning]->saturation);

  return(0);
}


