/* 
   CCD library, SBIG mutation

   wrapper functions to the low-level SBIG library.

   reimplements all functions to define architecture independent code model

   this is a thread library, use macro _REENTRANT in your code
   and link with pthread library: c89 -D_REENTRANT try.c -lccdsig -lpthread

   $Id: ccdsbig.c,v 1.1 2007-05-07 18:36:48 hroch Exp $

*/

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <pthread.h>
#include <time.h>
#include <sys/time.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <signal.h>
#include <syslog.h>
#include <fitsio.h>
#include <sbig.h>
#include "nightview.h"
#include "ccdtypes.h"
#include "ccd.h"
#include "ccdcommon.h"

/* debugging */
#define DEBUG_SBIG 0
#define DEBUG_EXTRA 1

/* HW defaults */
#define DEFAULT_DEVICE 0x378
#define BITPIX 16
#define BIAS 100.0

/* default observation site coordinates, values for MonteBoo,
   the values are real numbers which you want write to FITS header */
#define ALTITUDE 304.0
#define LONGITUDE 16.58395
#define LATITUDE 49.204128

/* realtime strategy, undef if you have a problems in image during readout */
#define REALTIME_RUN 1

/* info sbig */
static struct sbig_init info;

/* status sbig - to minimize communication over a link */
struct sbig_status status;

/*
#define NFILTERS 5
static char *filters[NFILTERS] = {NULL, NULL, NULL, NULL, NULL};
static int filter = -1;
*/

static struct timeval start_filter;
static float filter_time = 0.0;
static char *filter = NULL;

#define PTHREAD_NULL (pthread_t)0
static pthread_t thread_readout = PTHREAD_NULL;
static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;

/* pstatus indicates the state of the camera readout:
    the value -1.0 is for no image is available
    the value in range 0.0..100.0 readout in progress
    the value is greather then 100.0 image succesfully readed, wait for save
*/
static float pstatus = -1.0;

/* stops the readout when set to non-zero value */
static  int readout_stop;

/* the filename of the current downloaded image, non-null when pstatus>100.0 */
static char *fitsname = NULL;

/* name of object */
static char *objectname = NULL;

/* telescope's name */
static char *telename = NULL;

/* site's name */
static char *sitename = NULL;

/* site parametrs */
static double longit = 0.0;
static double lat = 0.0;
static double alt = 0.0;

/* exposure time */
static float exposure_time = -1.0;

/* indication of exposure status */
static int exp_progress = 0;

/* start of exposure in FITS format */
static char datetime[80];

/* telescope position at start od exposure */
static double tel_ra, tel_dec;

/* ---------------------------------------------------------------------- */

/* 

 This function initialize libary. Must be called before any other function.
 As argument needs device HW adrress, eg. address of the paralel port
 (= 0x378 for first, 0x3bc for second and 0x278 for third parallel port).
 The debug argument sets the verbosity level (0 status codes, 1 errors,
 .. 5 all). The default value (first parallel port and no nebug) is
 used when parameter is set to negative number.

 It set the global variable debug.

 Return initialized CCD structure (OK) or NULL (fails).

*/

int ccd_login(char *a)
{
  return(OK);
}

void *ccd_init(char *sdevice)
{
  CCD *ccd;
  int device;

  if( getuid() != 0 ) {
    syslog(LOG_NOTICE,"This binary must be run only the superuser.\n");
    return(NULL);
  }

  ccd = ccd_new(sdevice);

  if( ccd )
    syslog(LOG_NOTICE,"server started (address=0x%s)\n",sdevice);
  else
    syslog(LOG_ERR,"server not started. ccd not created\n");

  return((void *) ccd);
}

/* ------------------------------------------------------------------*/

static int devicename(char *a)
{
  if( strcmp(a,"LPT1")==0 || strcmp(a,"0x378")==0 || strcmp(a,"lp0")==0 )
    return(0x378);
  else if( strcmp(a,"LPT2")==0 || strcmp(a,"0x278")==0 || strcmp(a,"lp1")==0 )
    return(0x278);
  else if( strcmp(a,"LPT3")==0 || strcmp(a,"0x3BC")==0 || strcmp(a,"lp2")==0 )
    return(0x3BC);
  else
    return(DEFAULT_DEVICE);
}

/* ------------------------------------------------------------------*/


/* switch on camera, check the communication link, if the connection
   is created, the basic info about a camera model is readed, 
*/

int ccd_connect(CCD *ccd)
{
  struct sbig_control control;
  int ret;
  int device;

  if( ccd == NULL ) 
    return(FAIL);  

  control.fan_on = 1;
  control.shutter = SBIG_CCD_SHUTTER_OPENED;
  control.led = SBIG_LED_BLINK_SLOW;

  /* init camera */
  device = devicename(ccd_get_address(ccd));
  if( (ret = sbig_init(device,DEBUG_SBIG,&info)) != 0 || 
      (ret = sbig_control(&control)) != 0) {
    syslog(LOG_ERR,"%s\n", sbig_show_error(ret));
    ccd_set_on(ccd,0);
    return(FAIL); 
  }
  else {
    ccd_set_on(ccd,1);
    ccd_set_fan(ccd,1);
    
    sitename = ccd_get_site_name(ccd);
    telename = ccd_get_telescope_name(ccd);
    longit = ccd_get_site_long(ccd);
    lat = ccd_get_site_lat(ccd);
    alt = ccd_get_site_alt(ccd);
    return(OK);
  }
}

/* switch off camera */
int ccd_shutdown(CCD *ccd)
{
  if( ccd ) {
    temp_off(ccd);
    ccd_set_on(ccd,0);
    return(OK); }
  else
    return(FAIL);
} 


/*
   reinitialize info about the camera, the informations contains
   model information, as parametres get # of ccd (0 = imaging, 
   1 = tracking) and readout mode, this is an info for this ccd 
   and mode

 */

char *ccd_info(CCD *ccd, int chip, int mode)
{
  char *c;
  char *name;

  name = NULL;
  if( (name = malloc(64+1)) )
    strncpy(name, info.camera_name,64);
  else
    return(NULL);

  c = ccd_new_info(ccd,
		   chip,
		   name,
	       info.camera_info[chip].readout_mode[mode].width,
	       info.camera_info[chip].readout_mode[mode].height,
	       info.camera_info[chip].readout_mode[mode].pixel_width/1e3,
	       info.camera_info[chip].readout_mode[mode].pixel_height/1e3,
	       info.camera_info[chip].readout_mode[mode].gain/100.0);
  if( c == NULL )
    free(name);
  return(c);
}

/*-------------------------------------------------------------*/

int temp_init(CCD *ccd)
{
  if( ccd ) 
    return(OK);
  else
    return(FAIL);
}

/*-------------------------------------------------------------*/

float temp_ccd(CCD *ccd)
{
  if(ccd_get_on(ccd) ) {
    sbig_get_status(&status);
    return(0.1*status.ccd_temperature); 
  }
  else
    return(-999.9);
}

/*---------------------------------------------------------------*/

float temp_air(CCD *ccd)
{
  if( ccd_get_on(ccd) ) {
    sbig_get_status(&status);
    return(0.1*status.air_temperature); 
  }
  else
    return(-999.9);
}

/*---------------------------------------------------------------*/

float temp_regul(CCD *ccd)
{
  if( ccd_get_on(ccd) ) {
    sbig_get_status(&status);
    return(status.cooling_power/2.55); 
  }
  else
    return(0.0);
}

/*---------------------------------------------------------------*/
int temp_set(CCD *ccd, float temperature)
{
  struct sbig_cool cool;
  int ret;

  if( ccd_get_on(ccd) ) {
    cool.temperature = rint((double) 10.0*temperature);
    cool.regulation = 1;
    cool.direct_drive = 1;

    if( (ret = sbig_set_cooling(&cool)) != 0) {
      syslog(LOG_ERR,"%s\n", sbig_show_error(ret));
      return(FAIL); }
    else
      return(OK);
  }
  else
    return(FAIL);
}

/*---------------------------------------------------------------*/

int temp_off(CCD *ccd)
{
  struct sbig_cool cool;
  int ret;

  if( ccd_get_on(ccd) ) {
    cool.temperature = 0;
    cool.regulation = 0;
    cool.direct_drive = 0;

    if( (ret = sbig_set_cooling(&cool)) != 0) {
      syslog(LOG_ERR,"%s\n", sbig_show_error(ret));
      return(FAIL); }
    else
      return(OK);
  }
  else 
    return(FAIL);
}

/*-------------------------------------------------------*/

/* only server can call this routine during initialization phase*/

/*
int filter_define(char *filt[])
{
  int i;
  
  for( i = 0; i < NFILTERS; i++ ) {
    if( filters[i] ) free(filters[i]);
    if( filt[i] ) {
      filters[i] = malloc(strlen(filt[i])+1);
      strcpy(filters[i],filt[i]);
#ifdef DEBUG
  fprintf(stderr,"%s:%d: filter %d set to %s\n",__FILE__,__LINE__,i,filters[i]);
#endif 
    }
    else 
      filters[i] = NULL;
  }
  return(OK);
}
*/

int filter_init(CCD *ccd)
{
  if( ccd_get_on(ccd) )
    return(OK);
  else
    return(FAIL);
}

char *filter_list(CCD *ccd)
{
  char *line, *filter;
  int i,j,len,nfilters;

  nfilters = ccd_get_nfilters(ccd);
  len = 0;
  for(i = 0; i < nfilters; i++) {
    filter = ccd_get_filter(ccd,i);
    if( filter )
      len = len + strlen(filter) + 3;
  }

  if( (line = malloc(len+1)) ) {
    j = 0;
    for( i = 0; i < nfilters; i++) {
      filter = ccd_get_filter(ccd,i);
      if( filter ) {
	snprintf(line+j,len-j,"'%s' ",filter);
	j = j + strlen(filter) + 3;
      }
    }
    return(line);
  }
  else
    return(NULL);
}


int filter_set(CCD *ccd, char *filt)
{
  struct sbig_pulse pulse;
  int i, ret;
  int nfilters;

  if( ccd_get_on(ccd) == 0 ) 
    return(FAIL);
  
  if( filter_status(ccd) == RUN )
    return(FAIL);

  nfilters = ccd_get_nfilters(ccd);

  for(i = 0; i < nfilters; i++ ) {

    filter = ccd_get_filter(ccd,i);
    if( filter && filt && strstr(filter,filt) ) {

#ifdef DEBUG
      printf("%s %s %d %s\n",filter, filt, i, ccd_get_cfilter(ccd));
#endif
    
      /* current filter required - no action*/
      if( strstr(filter,ccd_get_cfilter(ccd))  )
         return(OK);

      /* init magic numbers for ST8 filter wheel */
      pulse.pulse_interval = 18270;
      pulse.nmbr_pulses = 60;
      pulse.pulse_width = 500 + 300*i;
      if( (ret = sbig_pulse(&pulse)) != 0) {
	syslog(LOG_ERR,"%s\n", sbig_show_error(ret));
	return(FAIL); }
      else {
	filter_time = (1500L*pulse.pulse_width)/1e6;
	gettimeofday(&start_filter,NULL);
	ccd_set_filter(ccd,i);
	return(OK);
      }
    }
  }
  return(FAIL);              /* required filter not found */
}

char *filter_get(CCD *ccd)
{
  return(ccd_get_cfilter(ccd));
}

int filter_status(CCD *ccd)
{
  struct timeval time;
  float sec;

  if( ccd_get_on(ccd) == 0) 
    return(EOE);
  else {
    gettimeofday(&time,NULL);
    sec = (time.tv_sec - start_filter.tv_sec) 
      + (time.tv_usec - start_filter.tv_usec)/1e6;
    if( sec <= filter_time )
      return(RUN);
    else
      return(IDLE);
  }
}


/*-------------------------------------------------------------*/

int exp_start(CCD *ccd, float exptime, int shutter, int chip)
{
  struct sbig_expose expose;
  struct sbig_control control;
  int ret;
  int timeref;

#ifdef DEBUG
  printf("expstart: %f %d %d\n",exptime,shutter,chip);
#endif

  if( ccd_get_on(ccd) == 0 )
    return(FAIL);

  if( !( 0.11 <= exptime && exptime < 167777.16) )
    return(FAIL);

  exposure_time = exptime;
  expose.ccd = chip;
  expose.exposure_time = rint((double) 100.0*exptime);
  expose.abg_state = SBIG_ABG_OFF;
  expose.shutter = shutter;

  control.fan_on = 1;
  control.shutter = SBIG_EXPOSE_SHUTTER_UNCHANGED;
  control.led = SBIG_LED_ON;

  /* save start time */
  ret = 0;
  fits_get_system_time(datetime,&timeref,&ret);
  if( timeref || ret ) 
    syslog(LOG_ERR,"It is not possible to get UTC, using local time.\n");

  /* save telescope position */
  ccd_get_telescope_ra_dec(ccd,&tel_ra, &tel_dec);

  if( (ret = sbig_expose(&expose)) != 0 || 
      (ret = sbig_control(&control)) != 0 ) {
    syslog(LOG_ERR,"%s\n", sbig_show_error(ret));
    exp_progress = 0;
    return(FAIL); 
  }
  else {
    exp_progress = 1;
    return(OK);
  }
}


int exp_stat(CCD *ccd, int chip)
{
  struct sbig_status status;
  int ret;

  if( ccd_get_on(ccd) == 0 ) 
    return(FAIL);

  if( ! exp_progress )
    return(FAIL);

  if( (ret = sbig_get_status(&status)) != 0 ) {
    syslog(LOG_ERR,"%s\n", sbig_show_error(ret));
    return(FAIL); }
  else {
    
    if( chip == 0 ) {
      if( status.imaging_ccd_status == 0 )
	return(IDLE);
      else if( status.imaging_ccd_status == 2 )
	return(RUN);
      else if( status.imaging_ccd_status == 3 )
	return(EOE);
      else
	return(FAIL);
    } 
    else if( chip == 1 ) {
      if( status.tracking_ccd_status == 0 )
	return(IDLE);
      else if( status.tracking_ccd_status == 2 )
	return(RUN);
      else if( status.tracking_ccd_status == 3 )
	return(EOE);
      else
	return(FAIL);
    }
    else
      return(FAIL);

  }
}

int exp_stop(CCD *ccd, int chip)
{
  exp_progress = 0;
  return(1);
}


/* ------------------------------------------------------------------- */

float read_stat(CCD *ccd)
{
  float x;

  if( ccd_get_on(ccd) == 0 ) 
    return(-1.0);

  pthread_mutex_trylock(&mutex);
  x = pstatus;
  pthread_mutex_unlock(&mutex);

#ifdef DEBUG_EXTRA  
  printf("pstatus=%f\n",pstatus);
#endif 

  /* clean thread */
  /* not need in detached state
  if( x > 99.9 ) {
    if( pthread_join(thread_readout, (void *) fitsname) )
      syslog(LOG_ERR,"Readout thread join: %m\n");
  }
  */

  return(x);
}

static int readout_callback(float p)
{
  int i;

  pthread_mutex_lock(&mutex);
  i = readout_stop;
  if( i )
    pstatus = p;
  else
    pstatus = -1.0;
    
#ifdef DEBUG_EXTRA
  fprintf(stderr,"%s:%d: callback: downloaded: %f%%\n",__FILE__,__LINE__,pstatus);
#endif 
  pthread_mutex_unlock(&mutex);
  return(i);
}

static void clean_child(int sig)
{
  pid_t cpid;
  int pids;

  cpid = wait(&pids);
  if( ! WIFEXITED(pids) )
    fprintf(stderr,"Child process: '/sbin/hwclock --hctosys' terminated abnormally. Time synchronisation lost.\n");
  signal(SIGCHLD,SIG_DFL);
}

static int pthread_create_detach(pthread_t *thread, pthread_attr_t *thattr,  void *(*start_routine)(void *), void *arg)
{
  pthread_attr_t attr;

  if( pthread_attr_init(&attr) != 0 || 
      pthread_attr_setdetachstate(&attr,PTHREAD_CREATE_DETACHED) != 0 ) {
    return(-1);
  }
  
  if( pthread_create(thread,&attr,start_routine, arg) )
    return(-1);

  pthread_attr_destroy(&attr);
  return(0);
}

static void destroy_readout(struct sbig_readout *readout)
{
  free(readout->data);
  free(readout);
}

static void *readout_func(void *readout1)
{
  int ret;
  struct sbig_readout *readout;
  struct sbig_control control;
  char *file;
  char *keep_image();
  pid_t pid;

  readout = (struct sbig_readout *) readout1;

  control.fan_on = 1;
  control.shutter = SBIG_EXPOSE_SHUTTER_UNCHANGED;
  control.led = SBIG_LED_BLINK_FAST;

  if( (ret = sbig_control(&control)) != 0 ) { /* blink LED rapidly */
    syslog(LOG_ERR,"%s\n", sbig_show_error(ret));
    destroy_readout(readout);
    pthread_exit(NULL);
  }

#ifdef REALTIME_RUN
  sbig_set_linux_strategy(SBIG_DEFAULT_STRATEGY | ~SBIG_LOCK_ALL);
#endif

  ret = sbig_readout(readout);
  
#ifdef REALTIME_RUN
  sbig_set_linux_strategy(SBIG_DEFAULT_STRATEGY);
#endif

  /* SBIG protocol lost time synchronisation, be sure that it is correct */
  /*
  if( pthread_create_detach(&time_sync, NULL,(void *) time_func, NULL) ){
    syslog(LOG_ERR,"The time not synchronized.: %m\n");
    free(rr.data);
    pthread_exit(NULL);
  }
  */
  pid = fork();
  if( pid < 0 )
    syslog(LOG_ERR,"Fork failed. The time not synchronized.: %m\n");
  else if( pid == 0 ) {
    if( execl("/sbin/hwclock","hwclock","--hctosys",NULL) < 0 )
      syslog(LOG_ERR,"execl hwclock: %m\n");
    exit(0);
  }
  else
    signal(SIGCHLD,clean_child);    

  if (ret < 0) {
    syslog(LOG_ERR,"%s\n", sbig_show_error(ret));
    destroy_readout(readout);
    pthread_exit(NULL);
  }

  control.led = SBIG_LED_BLINK_SLOW;
  if( (ret = sbig_control(&control)) != 0 ) { /* blink LED slowly */
    syslog(LOG_ERR,"%s\n", sbig_show_error(ret));
    destroy_readout(readout);
    pthread_exit(NULL);
  }

  if ( pstatus > 0.0 ) { 
     file = keep_image(readout->width, readout->height, readout->data, 
		    readout->ccd, readout->binning, readout->x, readout->y);

     pthread_mutex_lock(&mutex);
     if( (fitsname = malloc(strlen(file)+1)) ) {
       strcpy(fitsname,file);
     }
     pthread_mutex_unlock(&mutex);
  }

  destroy_readout(readout);
#ifdef DEBUG
  printf("terminating readout thread...\n");
#endif
  pthread_exit(NULL);
}


int read_start(CCD *ccd, int bitpix, int bin, int x1, int y1, int x2, int y2, int chip)
{
  struct sbig_readout *readout;
  int nx, ny, i;

  /* this is unsecure, needs create readout non-static in this function
    and destroy at the thread end not here!
  */

  if( ccd_get_on(ccd) == 0 )
    return(FAIL);

  /* check if radout is in progress */
  i = read_stat(ccd);
  if( 0.0 < i &&  i < 100.0 )
    return(FAIL);

  if( !( 0 < bin && bin <= 3 ) )
    return(FAIL);

  if( x1 < 0 || y1 < 0 )
    return(FAIL);

  nx = x2 - x1 + 1;
  ny = y2 - y1 + 1;

  if( nx <= 0 || ny <= 0 )
    return(FAIL);

  if( (readout = malloc(sizeof(struct sbig_readout))) == NULL ) {
    syslog(LOG_ERR,"There is not room for readout structure.\n");
    return(FAIL);
  }
  readout->ccd = chip;
  readout->binning = bin - 1;
  readout->x = x1 - 1;
  /*  readout->y = (info.camera_info[chip].readout_mode[0].height/bin - 1) - (y2 - y1);*/
  readout->y = y1 - 1;
  readout->width = nx;
  readout->height = ny;
  readout->data_size_in_bytes = readout->width*readout->height*(bitpix/8);
  readout->data = malloc(readout->data_size_in_bytes);
  readout->callback = readout_callback;
  if( readout->data == NULL ) {
    syslog(LOG_ERR,"There is no room for a new image. Needs %d bytes\n",readout->data_size_in_bytes);
    free(readout);
    return(FAIL);
  }

  pstatus = -1.0;
  readout_stop = 1;
  if( fitsname )
    free(fitsname);
  fitsname = NULL;

#ifdef DEBUG
  printf("detaching thread..%p\n",readout);
#endif
  if( pthread_create_detach(&thread_readout, NULL, (void *)readout_func, (void *) readout) ){
    syslog(LOG_ERR,"Readout thread not created: %m\n");
    destroy_readout(readout);
    return(FAIL);
  }
  return(OK);
}

int read_stop(CCD *ccd)
{
  pthread_mutex_lock(&mutex);
  readout_stop = 0;
  /*  pstatus = -1.0;*/
  pthread_mutex_unlock(&mutex);
  return(1);
}

char *read_download(CCD *ccd)
{
  char *file;

  file = NULL;

  pthread_mutex_lock(&mutex);
  if( ccd && pstatus >= 100.0 && fitsname && 
      (file = malloc(strlen(fitsname)+strlen("file://")+1)) ) {
    strcpy(file,"file://");
    strcat(file,fitsname);
    pstatus = -1.0;
  }
  pthread_mutex_unlock(&mutex);

  return(file);
}

int read_remove(CCD *ccd)
{

#ifdef DEBUG
  printf("Removing %s\n",fitsname);
#endif
  if( ccd && fitsname ) {
    if( unlink(fitsname) < 0 )
      syslog(LOG_ERR,"unlink: %m\n");
    free(fitsname);
    fitsname = NULL;
  }
  return(OK);
}


/* wrong way !!! need better synchro with mount drivers !!! this as old versiion*/
/*
void mount(float *ra, float *dec)
{
  FILE *f;

  *ra = 0.0;
  *dec = 0.0;
  if( (f = fopen("/home/juzr/.xmove.coo","r")) ) {
    fscanf(f,"%f",ra);
    fscanf(f,"%f",dec);
    fclose(f);
  }

}
*/


char *keep_image(int nx, int ny, unsigned short *fitsimage, int ccd, int binnig, 
		 int startx, int starty)
{
  fitsfile *fptr;       /* pointer to the FITS file, defined in fitsio.h */
  long naxes[2], npixels;
  char *output;
  float temp;
  int i, j, itmp, ret;

  /* the files are created read-only */
  umask( S_IWUSR | S_IWGRP | S_IWOTH );
  
  /* contvert SBIG image to FITS image array shape, flip,
     this still work on unsigned char readout.data masked to unsigned short */
  for( i = 0; i < ny/2; i++ ) 
    for( j = 0; j < nx; j++ ) {
      itmp = fitsimage[nx*i + j];
      fitsimage[nx*i + j] = fitsimage[nx*(ny - i - 1) + j];
      fitsimage[nx*(ny - i - 1) + j] = itmp;
    }

  /* BIAS correction, the BIAS is the constant and it's 100 ADU */
  /* The subtraction of offset from images is problematics operation. */ 
  /*
  for( i = 0; i < nx*ny; i++ )
    fitsimage[i] = fitsimage[i] - BIAS;
  */

  /* check and read mount coordinate file */
  /*  mount(&ra,&dec);*/

  /* save image to file */
  if( (output = tmpnam(NULL)) == NULL) {
    syslog(LOG_ERR,"Tmp file creation: %m\n");
  }

  ret = 0;         /* initialize status before calling fitsio routines */
  naxes[0] = nx;
  naxes[1] = ny;
  npixels  = naxes[0]*naxes[1];          /* number of pixels in the image */

  if (fits_create_file(&fptr, output, &ret))    /* create new FITS file */
    fits_report_error(stderr, ret);   /* print error if error occurs */
  if ( fits_create_img(fptr, USHORT_IMG, 2, naxes, &ret) )
    fits_report_error(stderr, ret);

  fits_update_key(fptr,TFLOAT,"EXPTIME",&exposure_time,"Exposure time (seconds)",&ret);
  fits_update_key(fptr,TSTRING,"DATE-OBS",datetime,"UTC of exposure start",&ret);
  fits_update_key(fptr,TSTRING,"FILTER",filter,"filter",&ret);

  /*
  if( objectname ) 
    fits_update_key(fptr,TSTRING,"OBJECT",objectname,"name of object",&ret);
  */

  if( tel_ra > -999.0 && tel_dec > -999.0 ) {
     temp = tel_ra;
     fits_update_key(fptr,TFLOAT,"MOUNTRA",&temp,"Approx. Right Ascension (deg)",&ret);
     temp = tel_dec;
     fits_update_key(fptr,TFLOAT,"MOUNTDEC",&temp,"Approx. Declination (deg)",&ret);
  }

  temp = 0.1*status.ccd_temperature;
  fits_update_key(fptr,TFLOAT,"TEMPERAT",&temp,"Camera temperature (C)",&ret);
  temp = 0.1*status.air_temperature;
  fits_update_key(fptr,TFLOAT,"TEMPOUT",&temp,"Air temperature (C)",&ret);

  startx = startx + 1;
  starty = starty + 1;
  fits_update_key(fptr,TSHORT,"XPOS",&startx,"X position of start pixel",&ret);
  fits_update_key(fptr,TSHORT,"YPOS",&starty,"Y position of start pixel",&ret);
  i = binnig + 1;
  fits_update_key(fptr,TSHORT,"XFACTOR",&i,"Camera x binning factor",&ret);
  fits_update_key(fptr,TSHORT,"YFACTOR",&i,"Camera y binning factor",&ret);  
  if( i == 1 ) 
    i = 16383;
  else
    i = 65535;
  fits_update_key(fptr,TUSHORT,"SATURATE",&i,"Saturation limit",&ret);
  fits_update_key(fptr,TSTRING,"CAMTYPE",info.camera_name,"Camera manufacturer and model",&ret);

  fits_update_key(fptr,TSTRING,"SITE",sitename,"Name of observatory",&ret);
  fits_update_key(fptr,TDOUBLE,"ALTITUDE",&alt,"Observatory altitude (meters)",&ret);
  fits_update_key(fptr,TDOUBLE,"LONGITUD",&longit,"Observatory longitude (deg) E+, W-",&ret);
  fits_update_key(fptr,TDOUBLE,"LATITUDE",&lat,"Observatory latitude (deg)",&ret);
  fits_update_key(fptr,TSTRING,"TELESCOP",telename,"Identifier of telescope",&ret);

  fits_write_comment(fptr,"This file was written by the Nightview package version:",&ret);
  fits_write_comment(fptr,NIGHTVER,&ret); 
  fits_write_comment(fptr,"Homepage: http://www.physics.muni.cz/mb/nightview/",&ret);

  if ( fits_write_img(fptr, TUSHORT, 1, npixels, fitsimage, &ret) )
    fits_report_error(stderr, ret);
  if ( fits_close_file(fptr, &ret) )
    fits_report_error(stderr, ret);

  /* the temporary files owned by nobody:nogroup */
  if( chown(output,65534,65534) )
    syslog(LOG_ERR,"chown: %m\n");

  if( ret ) {
    free(output);
    return(NULL);
  }
  else
    return(output);
}

int object_set(char *name)
{
  if( (objectname = malloc(strlen(name) + 1)) ) {
    strcpy(objectname,name);
    return(OK);
  }
  else
    return(FAIL);
}

