/* 
   CCD library, SBIG Universal Driver/Library wrapper functions

   this is a thread library, use macro _REENTRANT in your code
   and link with pthread library: c89 -D_REENTRANT try.c -lsbigudrv -lpthread

   $Id: sbiguni.c,v 1.14 2008-08-15 20:42:30 hroch Exp $

*/

/* TARGET is defined in sbigudrv.h */
#define TARGET 7

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <pthread.h>
#include <time.h>
#include <sys/time.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <signal.h>
#include <syslog.h>
#include <sbigudrv.h>
#include "nightview.h"
#include "ccdtypes.h"
#include "ccd.h"
#include "ccdcommon.h"
#include "exec.h"

/* debugging */
#define DEBUG_SBIG 1
#define DEBUG_EXTRA 1

/* HW defaults */
#define BITPIX 16

/* realtime strategy, undef if you have a problems in image during readout */
#define REALTIME_RUN 1

typedef struct {
  int chip;
  int mode;
  int x,y,width,height;
  int nsize;
  unsigned short *data;
  CCD *ccd;
} readout_par;

#define PTHREAD_NULL (pthread_t)0
static pthread_t thread_readout = PTHREAD_NULL;
static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;

/* pstatus indicates the state of the camera readout:
    the value -1.0 is for no image is available
    the value in range 0.0..100.0 readout in progress
    the value is greather then 100.0 image succesfully readed, wait for save
*/
static float pstatus = -1.0;

/* stops the readout when set to non-zero value */
static  int readout_stop;

static int debug = 10;

/* indication of autofreeze */
static int afreeze = 0;

/* buffered tempeartures during autofreeze */
static float temperature_ccd = -999.9; 
static float temperature_air = -999.9; 
static float temperature_setpoint = -999.9; 
static float temperature_regul = -999.9; 

/* number and string for error messages of CFW filter wheel */
#define NCFW 6
static char *cfwmsg[] = {"No Error", "CFW Busy", "Bad Command", 
			 "Calibration Error", "Motor Timeout", "Bad Model"};


/* -----------------------------------------------------------------------*/


/* auxiliar functions */

/* convert general description of device address to enum type */
static OpenDeviceParams devicename(char *a, OpenDeviceParams *x)
{
  int i,j,k,l;

  /* nullify the structure for sure */
  x->deviceType = DEV_NONE;
  x->lptBaseAddress = 0;
  x->ipAddress = 0;

  if( strcmp(a,"LPT1")==0 || strcmp(a,"0x378")==0 || strcmp(a,"lp0")==0 ){
    x->deviceType = DEV_LPT1;
    x->lptBaseAddress = 0x378;
  }  
  else if( strcmp(a,"LPT2")==0 || strcmp(a,"0x278")==0 || strcmp(a,"lp1")==0 ){
    x->deviceType = DEV_LPT2;
    x->lptBaseAddress = 0x278;
  }
  else if( strcmp(a,"LPT3")==0 || strcmp(a,"0x3BC")==0 || strcmp(a,"lp2")==0 ){
    x->deviceType = DEV_LPT3;
    x->lptBaseAddress = 0x3BC;
  }
  else if( strcmp(a,"USB")==0 || strcmp(a,"usb")==0 ){
    x->deviceType = DEV_USB;
  }
  else if( strcmp(a,"USB1")==0 || strcmp(a,"usb0")==0 ){
    x->deviceType = DEV_USB1;
  }
  else if( strcmp(a,"USB2")==0 || strcmp(a,"usb1")==0 ){
    x->deviceType = DEV_USB2;
  }
  else if( strcmp(a,"USB3")==0 || strcmp(a,"usb2")==0 ){
    x->deviceType = DEV_USB3;
  }
  else if( strcmp(a,"USB4")==0 || strcmp(a,"usb3")==0 ){
    x->deviceType = DEV_USB4;
  }
  else if( sscanf(a,"%d.%d.%d.%d",&i,&j,&k,&l) == 4 ){
    x->deviceType = DEV_ETH;
    x->ipAddress = 256*(256*(256*i + j) + k) + l; /* IP4 only !*/
  }    

  if( debug )
    syslog(LOG_INFO,"SBIG Device address: '%s' -> 0x%x 0x%x %ld\n",a,
           x->deviceType,x->lptBaseAddress,x->ipAddress);     
 
  return(*x);
}

/* wrapper for SBGUnivDrvCommand */
static int sbu(short Command, void *Parameters, void *Results)
{
  GetErrorStringParams epar;
  GetErrorStringResults eres;
  int j;

  epar.errorNo = SBIGUnivDrvCommand(Command,Parameters,Results);
  j = epar.errorNo == CE_NO_ERROR;

  if( ! j ) {
    SBIGUnivDrvCommand(CC_GET_ERROR_STRING,&epar,&eres);
    syslog(LOG_ERR,"SBIGUniDrvError(#%d): %s\n",epar.errorNo,eres.errorString);

#ifdef DEBUG_SBIG
    fprintf(stderr,"SBIGUniDrvError(#%d): %s (command:%d)\n",
	    epar.errorNo,eres.errorString,Command);
#endif
  }

  return(j);
}


/* convert: packed bcd number to long */

static unsigned long bcd2long(unsigned long a)
{
   int i,n[2*sizeof(long)], nl;
   long b;

   nl = 2*sizeof(long);

   for(i = 0; i < nl; i++) {

     n[i] = a % 16;
     a = a >> 4;
     /*printf("%d %d %d\n",i,a,n[i]);*/
  }
  b = 0;
  for( i = nl-1; i >= 0 ; i-- )
    b = 10*b + n[i];

  return b;
}

static float ad2t(float ad, float r_bridge, float max_ad, float r_ratio, 
		  float r0, float t0, float dt)
{
  float r,t;

  r = r_bridge/(max_ad/ad - 1.0);
  t = t0 - dt*(log(r/r0)/log(r_ratio));

  return(t);
}

static float t2ad(float t, float r_bridge, float max_ad, float r_ratio, 
		  float r0, float t0, float dt)
{
  float r,ad;

  r = r0*exp((log(r_ratio)*(t0 - t))/dt);
  ad = max_ad/(r_bridge/r + 1.0);

  return(ad);
}



/* ---------------------------------------------------------------------- */


int ccd_login(char *a)
{
  return(OK);
}

/* 

 This function initialize libary. Must be called before any other function.
 As argument needs device HW adrress, eg. address of the paralel port
 (= 0x378 for first, 0x3bc for second and 0x278 for third parallel port).
 The debug argument sets the verbosity level (0 status codes, 1 errors,
 .. 5 all). The default value (first parallel port and no nebug) is
 used when parameter is set to negative number.

 It set the global variable debug.

 Return initialized CCD structure (OK) or NULL (fails).

*/

CCD *ccd_init(char *sdevice)
{
  CCD *ccd;

  /*
  if( getuid() != 0 ) {
    syslog(LOG_NOTICE,"This binary must be run only by the superuser.\n");
    return(NULL);
  }
  */ 

  if( ! sdevice ) {
     syslog(LOG_INFO,"Device address not specified. Using 'usb0'.");
     sdevice = strdup("usb0");
  }

  /* new ccd instance */
  if( (ccd = ccd_new(sdevice)) ) {
    syslog(LOG_NOTICE,"Server setup succesfull (device=%s).\n",sdevice);
    return(ccd);
  }
  else {
    syslog(LOG_ERR,"Server setup failed.\n");
    return(NULL);
  }
}

/* switch on camera, check the communication link, if the connection
   is created, the basic info about a camera model is readed, 
*/

int ccd_connect(CCD *ccd)
{
  GetDriverInfoParams dpar;
  GetDriverInfoResults0 dres;
  EstablishLinkParams linkp;
  EstablishLinkResults linkr;
  GetLinkStatusResults links;
  MiscellaneousControlParams misc;
  OpenDeviceParams device;
  int i,j,k;

  if( ! ccd ) 
    return(FAIL);  

  if( ccd_get_on(ccd) && ccd_connected(ccd) )
    return(OK);

  linkp.sbigUseOnly = 0;
  misc.fanEnable = TRUE;
  misc.shutterCommand = SC_OPEN_SHUTTER;
  misc.ledState = LED_BLINK_LOW;

  /* specify devicename */
  device = devicename(ccd_get_address(ccd),&device);

  /* Open driver */
  i = sbu(CC_OPEN_DRIVER, NULL, NULL);

  if ( i && debug ) {

    /* ----------------------------------
       Query USB has malfunction (!), 
       if it is commented out, the connection to camera
       (via CC_EXTABLISHED_LINK) falls.
       ----------------------------------
    */
       

    /* Query USB */
    /*
    QueryUSBResults us;
    sbu(CC_QUERY_USB,NULL,&us);
    syslog(LOG_INFO,"Total number of cameras found = %d\n",us.camerasFound);
    int k;
    for( k = 0; k < us.camerasFound; k++) {
      syslog(LOG_INFO,"#%d:CAMERA_TYPE=%d\n",k,us.usbInfo[k].cameraType);
      syslog(LOG_INFO,"#%d:%s\n",k,us.usbInfo[k].name);
      syslog(LOG_INFO,"#%d:Serial number: %s\n",k,us.usbInfo[k].serialNumber);
    }
    */

    /* Get driver info */
    dpar.request = 0;
    sbu(CC_GET_DRIVER_INFO,&dpar,&dres);
    syslog(LOG_INFO,"SBIG Universal Driver Library info: %d %s %d\n",
	   dres.version,dres.name,dres.maxRequest);

    /* for future version of driver */
    /*
    dpar.request = 1;
    sbu(CC_GET_DRIVER_INFO,&dpar,&dres);
    syslog(LOG_INFO,"SBIG Universal Driver Library low level: %d %s %d\n",
	   dres.version,dres.name,dres.maxRequest);
    dpar.request = 2;
    sbu(CC_GET_DRIVER_INFO,&dpar,&dres);
    syslog(LOG_INFO,"SBIG Universal Driver Library USB loader: %d %s %d\n",
	   dres.version,dres.name,dres.maxRequest);
    */  
  }

  /* Open device */
  if( i ) 
    j = sbu(CC_OPEN_DEVICE,&device,NULL);
  else
    j = 0;

  /* Establish link */
  if( i && j )
    k = sbu(CC_ESTABLISH_LINK,&linkp,&linkr);
  else
    k = 0;

  if( i && j && k ) {

    /* Get link status */
    if( sbu(CC_GET_LINK_STATUS,NULL,&links) && links.linkEstablished == TRUE )
      syslog(LOG_INFO,"Link established to %d (par. port), %d (cam. type), #chanells %ld/%ld (total/failed)\n",
	     links.baseAddress,links.cameraType,links.comTotal,links.comFailed);
  }
  else
    syslog(LOG_WARNING,"Link to a camera can't be established.\n");

  if( i && j && k) {

    /* Setup miscellaneous functions */
    sbu(CC_MISCELLANEOUS_CONTROL,&misc,NULL);

    filter_connect(ccd);
    
    ccd_set_on(ccd,1);
    ccd_set_fan(ccd,1);

    ccd_info_all(ccd);
    ccd_set_filter(ccd,0);
    /*
    sitename = ccd_get_site_name(ccd);
    telename = ccd_get_telescope_name(ccd);
    longit = ccd_get_site_long(ccd);
    lat = ccd_get_site_lat(ccd);
    alt = ccd_get_site_alt(ccd);
    */
    return(OK);
  }
  else {

    if( j ) 
      sbu(CC_CLOSE_DEVICE,NULL,NULL);
    if( i )
      sbu(CC_CLOSE_DRIVER,NULL,NULL);

    ccd_set_on(ccd,0);
    return(FAIL); 
  }
}

int ccd_connected(CCD *ccd)
{
  GetLinkStatusResults links;
  int j;

  /* Get link status */
  j = sbu(CC_GET_LINK_STATUS,NULL,&links);
  return(links.linkEstablished == TRUE);
}

/* switch off camera */
int ccd_shutdown(CCD *ccd)
{
  MiscellaneousControlParams misc;

  misc.fanEnable = FALSE;
  misc.shutterCommand = SC_CLOSE_SHUTTER;
  misc.ledState = LED_OFF;

  if( ccd && ccd_get_on(ccd) ) {
    filter_disconnect(ccd);
    temp_off(ccd);
    ccd_set_on(ccd,0);
    sbu(CC_MISCELLANEOUS_CONTROL,&misc,NULL);
    sbu(CC_CLOSE_DEVICE,NULL,NULL);
    sbu(CC_CLOSE_DRIVER,NULL,NULL);

    syslog(LOG_ERR,"CCD camera on address %s shutdown.\n",ccd_get_address(ccd));
    /* ccd_free(ccd); */
    return(OK); }
  else
    return(FAIL);
}


/*
   reinitialize info about the camera, the informations contains
   model information, as parametres get # of ccd (0 = imaging, 
   1 = tracking) and readout mode, this is an info for this ccd 
   and mode

 */

CCDINFO *ccd_info(CCD *ccd, int chip, int mode)
{
  if( ccd )
    return(ccd->ccdinfo[0]);
  else
    return(NULL);
}

int ccd_info_all(CCD *ccd)
{
  ReadOffsetParams opar;
  ReadOffsetResults ores;
  GetCCDInfoParams p;
  GetCCDInfoResults0 r;
  GetCCDInfoResults2 r2;
  GetCCDInfoResults3 r3;
  int i,j, k, n;
  int width, height;
  float gain,pixelWidth,pixelHeight;
  int nbad,badc[MAXBAD];
  double minexp, maxexp;
  char firmware[11];
  int bitpix;

  /* default to 16 */
  bitpix = 16;

  if( ! ccd || ! ccd_get_on(ccd) )
    return(0);

  /* imaging chip */
  n = 0;
  p.request = CCD_INFO_IMAGING;
  i = sbu(CC_GET_CCD_INFO,&p,&r);

  if( i ) {

    snprintf(firmware,10,"%5.2f",bcd2long((long)r.firmwareVersion)/100.0);

    opar.ccd = n;
    sbu(CC_READ_OFFSET,&opar,&ores);

    if( r.cameraType == ST2K_CAMERA ) 
      minexp = 0.01;
    else
      minexp = 0.12;
    maxexp = 167777.16;

    width = r.readoutInfo[0].width;
    height = r.readoutInfo[0].height;
    pixelWidth = bcd2long(r.readoutInfo[0].pixelWidth)/100.0;
    pixelHeight = bcd2long(r.readoutInfo[0].pixelHeight)/100.0;
    gain = bcd2long((long) r.readoutInfo[0].gain)/100.0;

    if( r.cameraType != ST237_CAMERA ) {

      /* all cameras except PixCel 237/255 */
      p.request = CCD_INFO_EXTENDED;
      j = sbu(CC_GET_CCD_INFO,&p,&r2);

      if( j ) {

	nbad = r2.badColumns;
	for( k = 0; k < nbad; k++)
	  badc[k] = r2.columns[k];
	/* stupid but required due to type comaptibility int <-> short */
	
	ccd_new_info(ccd,n,r.name,firmware,r2.serialNumber,width,height,
		     ores.offset,bitpix,r2.imagingABG,pixelWidth,pixelHeight,
		     gain,r.readoutModes,minexp,maxexp,nbad,badc);
      }
    }
    else {

      /* Case of PixCel 237/255 */
      p.request = CCD_INFO_EXTENDED_5C;
      j = sbu(CC_GET_CCD_INFO,&p,&r3);

      if( gain < 1.0 ) bitpix = 12;
      if( j )

	ccd_new_info(ccd,n,r.name,firmware,NULL,width,height,ores.offset,
		     r3.adSize,0,pixelWidth,pixelHeight,gain,r.readoutModes,
		     minexp,maxexp,0,NULL);

    }

    if( ccd && ccd->ccdinfo[n] ) {

      for( k = 0; k < r.readoutModes && k < ccd->ccdinfo[n]->maxreadout; k++) {

	ccd_new_readout(ccd->ccdinfo[n],k,r.readoutInfo[k].mode,
			r.readoutInfo[k].width,r.readoutInfo[k].height,
			bcd2long((long) r.readoutInfo[k].gain)/100.0,
			bcd2long(r.readoutInfo[k].pixelWidth)/100.0,
			bcd2long(r.readoutInfo[k].pixelHeight)/100.0);
	
#ifdef DEBUG_EXTRA
	printf("%d %dx%d %f %f %f\n",k,r.readoutInfo[k].width,
	       r.readoutInfo[k].height,
	       bcd2long((long) r.readoutInfo[k].gain)/100.0,
	       bcd2long(r.readoutInfo[k].pixelWidth)/100.0,
	       bcd2long(r.readoutInfo[k].pixelHeight)/100.0);
#endif
      }
    }
    printf("out0:%s %dx%d\n",ccd->ccdinfo[0]->camera_name,
	   ccd->ccdinfo[0]->height,ccd->ccdinfo[0]->width);
  }

  /* tracking chip */
  n = 1;
  p.request = CCD_INFO_TRACKING;
  i = sbu(CC_GET_CCD_INFO,&p,&r);

  if( i ) {

    snprintf(firmware,10,"%5.2f",bcd2long((long)r.firmwareVersion)/100.0);
    printf("%s\n",firmware);

    opar.ccd = n;
    sbu(CC_READ_OFFSET,&opar,&ores);
    printf("%d\n",ores.offset);

    minexp = 0.12;
    maxexp = 655.35;

    width = r.readoutInfo[0].width;
    height = r.readoutInfo[0].height;
    pixelWidth = bcd2long(r.readoutInfo[0].pixelWidth)/100.0;
    pixelHeight = bcd2long(r.readoutInfo[0].pixelHeight)/100.0;
    gain = bcd2long((long) r.readoutInfo[0].gain)/100.0;

    bitpix = 8;
    printf("%d %d %f %f %f\n",width,height,pixelWidth,pixelHeight,gain);
    ccd_new_info(ccd,n,r.name,firmware,NULL,width,height,ores.offset,
		 bitpix,0,pixelWidth,pixelHeight,gain,r.readoutModes,
		 minexp,maxexp,0,NULL);

    printf("%s\n",ccd->ccdinfo[n]->camera_name);
    if( ccd && ccd->ccdinfo[n] ) {

      for( k = 0; k < r.readoutModes && k < ccd->ccdinfo[n]->maxreadout; k++) {

	ccd_new_readout(ccd->ccdinfo[n],k,r.readoutInfo[k].mode,
			r.readoutInfo[k].width,r.readoutInfo[k].height,
			bcd2long((long) r.readoutInfo[k].gain)/100.0,
			bcd2long(r.readoutInfo[k].pixelWidth)/100.0,
			bcd2long(r.readoutInfo[k].pixelHeight)/100.0);

#ifdef DEBUG_EXTRA
	printf("%d %dx%d %f %f %f\n",k,r.readoutInfo[k].width,
	       r.readoutInfo[k].height,
	       bcd2long((long) r.readoutInfo[k].gain)/100.0,
	       bcd2long(r.readoutInfo[k].pixelWidth)/100.0,
	       bcd2long(r.readoutInfo[k].pixelHeight)/100.0);
#endif
      }
    }
    printf("out1:%s %dx%d\n",ccd->ccdinfo[1]->camera_name,
	   ccd->ccdinfo[0]->height,ccd->ccdinfo[0]->width);
  }

  if( ccd && ccd->ccdinfo[0] )
    return(1);
  else
    return(0);
}

/*-------------------------------------------------------------*/

int temp_init(CCD *ccd)
{
  if( ccd ) 
    return(OK);
  else
    return(FAIL);
}

/*-------------------------------------------------------------*/

float temp_ccd(CCD *ccd)
{
  QueryTemperatureStatusResults t;

  if( ccd_get_on(ccd) ) {

    if( afreeze )
      return(temperature_ccd);

    sbu(CC_QUERY_TEMPERATURE_STATUS,NULL,&t);
    temperature_ccd = ad2t(t.ccdThermistor,10.0,4096.0,2.57,3.0,25.0,25.0);
    return(temperature_ccd);

  }
  else
    return(-999.9);
}

/*---------------------------------------------------------------*/

float temp_setpoint(CCD *ccd)
{
  QueryTemperatureStatusResults t;

  if( ccd_get_on(ccd) ) {

    if( afreeze )
      return(temperature_setpoint);

    sbu(CC_QUERY_TEMPERATURE_STATUS,NULL,&t);
    temperature_setpoint = ad2t(t.ccdSetpoint,10.0,4096.0,2.57,3.0,25.0,25.0);
    return(temperature_setpoint);
  }
  else
    return(-999.9);
}

/*---------------------------------------------------------------*/

float temp_air(CCD *ccd)
{
  QueryTemperatureStatusResults t;

  if( ccd_get_on(ccd) ) {

    if( afreeze )
      return(temperature_air);

    sbu(CC_QUERY_TEMPERATURE_STATUS,NULL,&t);
    temperature_air = ad2t(t.ambientThermistor,3.0,4096.0,7.791,3.0,25.0,45.0);
    return(temperature_air);
  }
  else
    return(-999.9);
}

/*---------------------------------------------------------------*/

float temp_regul(CCD *ccd)
{
  QueryTemperatureStatusResults t;

  if( ccd_get_on(ccd) ) {

    if( afreeze )
      return(temperature_regul);

    sbu(CC_QUERY_TEMPERATURE_STATUS,NULL,&t);
    temperature_regul = t.power/2.56;
    return(temperature_regul); 
  }
  else
    return(0.0);
}

/*---------------------------------------------------------------*/
int temp_set(CCD *ccd, float temperature)
{
  SetTemperatureRegulationParams t;

  if( ccd_get_on(ccd) && afreeze == 0 ) {

    t.regulation = REGULATION_ON;
    // t.regulation = REGULATION_ENABLE_AUTOFREEZE;
    t.ccdSetpoint = t2ad(temperature,10.0,4096.0,2.57,3.0,25.0,25.0);

    return(sbu(CC_SET_TEMPERATURE_REGULATION,&t,NULL));
  }
  else
    return(FAIL);
}

/*---------------------------------------------------------------*/

int temp_off(CCD *ccd)
{
  SetTemperatureRegulationParams t;

  if( ccd_get_on(ccd) && afreeze == 0 ) {

    t.regulation = REGULATION_OFF;
    return(sbu(CC_SET_TEMPERATURE_REGULATION,&t,NULL));
  }
  else 
    return(FAIL);
}

/*-------------------------------------------------------*/

/* only server can call this routine during initialization phase*/

/*
int filter_define(char *filt[])
{
  int i;
  
  for( i = 0; i < NFILTERS; i++ ) {
    if( filters[i] ) free(filters[i]);
    if( filt[i] ) {
      filters[i] = malloc(strlen(filt[i])+1);
      strcpy(filters[i],filt[i]);
#ifdef DEBUG
  fprintf(stderr,"%s:%d: filter %d set to %s\n",__FILE__,__LINE__,i,filters[i]);
#endif 
    }
    else 
      filters[i] = NULL;
  }
  return(OK);
}
*/

int filter_connect(CCD *ccd)
{
  CFWParams fpar;
  CFWResults fres;
  int i;

  if( ccd ) {

    fpar.cfwModel = CFWSEL_AUTO;
    fpar.cfwParam1 = 0;
    fpar.cfwParam2 = 0;
    fpar.outLength = 0;
    fpar.outPtr = NULL;
    fpar.inLength = 0;
    fpar.inPtr = NULL;

    fpar.cfwCommand = CFWC_OPEN_DEVICE;
    i = sbu(CC_CFW,&fpar,&fres);

    /* the CFWG_FIRMWARE_VERSION issue SBIGUniDrvError(#6): Bad Parameter*/
    fpar.cfwCommand = CFWC_GET_INFO;
    fpar.cfwParam1 = CFWG_FIRMWARE_VERSION;
    if( i && sbu(CC_CFW,&fpar,&fres) ) {
      syslog(LOG_INFO,"CFW: model %d, firmware %ld, pos#: %ld\n",
	     fres.cfwModel,fres.cfwResult1,fres.cfwResult2);
      syslog(LOG_INFO,"CFW: status %d, error %d, position #%d\n",
	     fres.cfwStatus,fres.cfwError,fres.cfwPosition);

      return(OK);
    }
    else {
      syslog(LOG_ERR,"CFW model: Connection to filter wheel failed.\n");
      return(FAIL);
    }
  }    
  else
    return(FAIL);
}

void filter_disconnect(CCD *ccd)
{
  CFWParams fpar;
  CFWResults fres;

  if( ccd ) {

    fpar.cfwModel = CFWSEL_AUTO;
    fpar.cfwParam1 = 0;
    fpar.cfwParam2 = 0;
    fpar.outLength = 0;
    fpar.outPtr = NULL;
    fpar.inLength = 0;
    fpar.inPtr = NULL;

    fpar.cfwCommand = CFWC_CLOSE_DEVICE;
    sbu(CC_CFW,&fpar,&fres);
  }
}



int filter_init(CCD *ccd)
{
  CFWParams fpar;
  CFWResults fres;

  if( ccd_get_on(ccd) ) {

    fpar.cfwModel = CFWSEL_AUTO;
    fpar.cfwParam1 = 0;
    fpar.cfwParam2 = 0;
    fpar.outLength = 0;
    fpar.outPtr = NULL;
    fpar.inLength = 0;
    fpar.inPtr = NULL;

    fpar.cfwCommand = CFWC_INIT;
    
    if( sbu(CC_CFW,&fpar,&fres) ) {
      syslog(LOG_INFO,"CFW model: %d successfuly initialized.\n",
	     fres.cfwModel);
      
      ccd_set_filter(ccd,0);
      return(OK);
    }
    else {
      if( 0 < fres.cfwError && fres.cfwError < NCFW )
	syslog(LOG_ERR,"Filter Init command failed with an error: %s (%d)\n",
	       cfwmsg[fres.cfwError],fres.cfwError);
      else
	syslog(LOG_ERR,"Filter Init command failed with an unknown error.\n");
      return(FAIL);
    }
  }
  else
    return(FAIL);
}



int filter_set(CCD *ccd, char *filt)
{
  CFWParams fpar;
  CFWResults fres;
  int i;
  int nfilters;
  char *filter;

  if( ccd_get_on(ccd) == 0 ) 
    return(FAIL);

  if( ! ccd_get_filterwheel(ccd)  && filt ) {
    ccd_set_filters(ccd,0,1,&filt);
    ccd_set_filter(ccd,0);
    return(OK);
  }
  
  if( filter_status(ccd) == RUN )
    return(FAIL);

  nfilters = ccd_get_nfilters(ccd);

  for(i = 0; i < nfilters; i++ ) {

    filter = ccd_get_filter(ccd,i);
    if( filter && filt && strstr(filter,filt) ) {

#ifdef DEBUG
      printf("%s %s %d %s\n",filter, filt, i, ccd_get_cfilter(ccd));
#endif
    
      /* current filter required - no action*/
      if( strstr(filter,ccd_get_cfilter(ccd))  )
         return(OK);

      fpar.cfwModel = CFWSEL_AUTO;
      fpar.cfwCommand = CFWC_GOTO;
      fpar.cfwParam1 = i+1;

      if( sbu(CC_CFW,&fpar,&fres) ) {
	ccd_set_filter(ccd,i);
	return(OK);
      }
      else {
	if( 0 < fres.cfwError && fres.cfwError < NCFW )
	  syslog(LOG_ERR,"Filter Goto command fails with an error: %s (%d)\n",
		 cfwmsg[fres.cfwError],fres.cfwError);
	else
	  syslog(LOG_ERR,"Filter Goto command fails with an unknown error.\n");
	return(FAIL);
      }
    }
  }
  return(FAIL);              /* required filter not found */
}

char *filter_list(CCD *ccd)
{
  return(ccd_filter_list(ccd));
}

char *filter_get(CCD *ccd)
{
  return(ccd_get_cfilter(ccd));
}

int filter_status(CCD *ccd)
{
  CFWParams fpar;
  CFWResults fres;

  if( ccd_get_on(ccd) == 0) 
    return(EOE);
  else {

    if( ! ccd_get_filterwheel(ccd) )
      return(IDLE);

    fpar.cfwModel = CFWSEL_AUTO;
    fpar.cfwCommand = CFWC_QUERY;

    if( sbu(CC_CFW,&fpar,&fres) ) {
      
      if( fres.cfwStatus == CFWS_BUSY )
	return(RUN);
      else if( fres.cfwStatus == CFWS_IDLE )
	return(IDLE);
      else
	return(EOE);

    }
    else {

      if( 0 < fres.cfwError && fres.cfwError < NCFW )
	syslog(LOG_ERR,"Filter Query command fails with a error: %s (%d)\n",
	       cfwmsg[fres.cfwError],fres.cfwError);
      else
	syslog(LOG_ERR,"Filter Query command fails with an unknown error.\n");

      return(EOE);
    }
  }
}


/*-------------------------------------------------------------*/

int exp_start(CCD *ccd, double exptime, int shutter, int chip)
{
  StartExposureParams epar;
  MiscellaneousControlParams misc;
  char datetime[81];
  double tel_ra, tel_dec;

#ifdef DEBUG
  printf("expstart: %f %d %d\n",exptime,shutter,chip);
#endif

  if( ccd_get_on(ccd) == 0 )
    return(FAIL);

  if( !( ccdinfo_get_minexp(ccd,chip) <= exptime && 
	 exptime < ccdinfo_get_maxexp(ccd,chip)) )
    return(FAIL);

  if( exp_stat(ccd,chip) == RUN )
    return(FAIL);

  epar.ccd = chip;
  epar.exposureTime = rint((double) 100.0*exptime);
  epar.abgState = ABG_LOW7;
  epar.openShutter = shutter;

  misc.fanEnable = TRUE;
  misc.shutterCommand = SC_LEAVE_SHUTTER;
  misc.ledState = LED_ON;

  /* save temperature */
  ccd_set_tempccd(ccd,temp_ccd(ccd));
  ccd_set_tempair(ccd,temp_air(ccd));

  /* save telescope position */
  ccd_get_telescope_ra_dec(ccd,&tel_ra, &tel_dec);
  ccd_set_ra(ccd,tel_ra);
  ccd_set_dec(ccd,tel_dec);

  /* save start time */
  systime(datetime);
  ccd_set_datetime(ccd,datetime);
  ccd_set_exptime(ccd,exptime);

  if( sbu(CC_START_EXPOSURE,&epar,NULL) && 
      sbu(CC_MISCELLANEOUS_CONTROL,&misc,NULL) )
    return(OK); 
  else 
    return(FAIL);
}


int exp_stat(CCD *ccd, int chip)
{
  QueryCommandStatusParams qpar;
  QueryCommandStatusResults qres;
  
  int imaging,tracking;
  char *a;

  if( ccd_get_on(ccd) == 0 ) 
    return(FAIL);

  qpar.command = CC_START_EXPOSURE;
  if( ! sbu(CC_QUERY_COMMAND_STATUS,&qpar,&qres) )
    return(FAIL);

  else {
    a = (void *) &(qres.status);
    imaging = *a;
    tracking = *(a+1);

    if( chip == 0 ) {
      if( imaging == 0 )
	return(IDLE);
      else if( imaging == 2 )
	return(RUN);
      else if( imaging == 3 )
	return(EOE);
      else
	return(FAIL);
    } 
    else if( chip == 1 ) {
      if( tracking == 0 )
	return(IDLE);
      else if( tracking == 2 )
	return(RUN);
      else if( tracking == 3 )
	return(EOE);
      else
	return(FAIL);
    }
    else
      return(FAIL);

  }
}

int exp_stop(CCD *ccd, int chip)
{
  EndExposureParams epar;

  epar.ccd = chip;
  return(sbu(CC_END_EXPOSURE,&epar,NULL));
}


/* ------------------------------------------------------------------- */

float read_stat(CCD *ccd)
{
  float x;

  if( ! ccd_get_on(ccd) ) 
    return(-1.0);

  pthread_mutex_trylock(&mutex);
  x = pstatus;
  pthread_mutex_unlock(&mutex);

#ifdef DEBUG_EXTRA  
  printf("pstatus=%f\n",pstatus);
#endif 

  return(x);
}


static int pthread_create_detach(pthread_t *thread, pthread_attr_t *thattr,
				 void *(*start_routine)(void *), void *arg)
{
  pthread_attr_t attr;

  if( pthread_attr_init(&attr) != 0 || 
      pthread_attr_setdetachstate(&attr,PTHREAD_CREATE_DETACHED) != 0 ) {
    return(-1);
  }
  
  if( pthread_create(thread,&attr,start_routine, arg) )
    return(-1);

  pthread_attr_destroy(&attr);
  return(0);
}

static void clean_child(int sig)
{
  pid_t cpid;
  int pids;

  cpid = wait(&pids);
  if( ! WIFEXITED(pids) )
    syslog(LOG_ERR,"Child process: '/sbin/hwclock --hctosys' terminated abnormally. Time synchronisation lost.\n");
  /*signal(SIGCHLD,SIG_DFL);*/
  exit(0);
}

static void time_synchro()
{
  pid_t pid;

  pid = fork();
  if( pid < 0 )
    syslog(LOG_ERR,"Fork failed. The time not synchronized.: %m\n");
  else if( pid == 0 ) {
    if( execl("/sbin/hwclock","hwclock","--hctosys",NULL) < 0 )
      syslog(LOG_ERR,"execl hwclock: %m\n");
    exit(0);
  }
  else
    signal(SIGCHLD,clean_child);    

}

static readout_par *create_readout(CCD *ccd,int chip, int mode,int x, int y,
				   int nx, int ny)
{
  readout_par *readout;
 
  if( (readout = malloc(sizeof(readout_par))) == NULL ) {
    syslog(LOG_ERR,"There is not room for readout structure.\n");
    return(NULL);
  }

  readout->ccd = ccd;
  readout->chip = chip;
  readout->mode = mode;
  readout->x = x;
  readout->y = y;
  readout->width = nx;
  readout->height = ny;
  readout->nsize = sizeof(unsigned short)*nx*ny;
  readout->data = malloc(readout->nsize);
  if( readout->data == NULL ) {
    syslog(LOG_ERR,"There is no room for a new image. Needs %d bytes\n",
	   readout->nsize);
    free(readout);
    return(NULL);
  }
  return(readout);
}

static void destroy_readout(readout_par *readout)
{
  if( readout ) 
    free(readout->data);
  free(readout);
  readout = NULL;
}

static void *readout_func(void *readout1)
{
  MiscellaneousControlParams misc;
  SetTemperatureRegulationParams t;
  StartReadoutParams spar;
  ReadoutLineParams lpar;
  EndReadoutParams epar;
  readout_par *readout;
  int i, binning;
  char *output;
  int m;
  /*
  char *device;
  const char *dev[] = {"LPT1","0x378","lp0","LPT2","0x278","lp1",
		       "LPT3","0x3BC","lp2", NULL};
  */
  readout = (readout_par *) readout1;

  misc.fanEnable = TRUE;
  misc.shutterCommand = SC_LEAVE_SHUTTER;
  misc.ledState = LED_BLINK_HIGH;

#ifdef DEBUG
  printf("setting up LED to blink rapidly...\n");
#endif
  if( ! sbu(CC_MISCELLANEOUS_CONTROL,&misc,NULL) ) { /* blink LED rapidly */
    destroy_readout(readout);
    pthread_exit(NULL);
  }

#ifdef DEBUG
  printf("Autofreeze...ON \n");
#endif
  t.regulation = REGULATION_FREEZE;
  sbu(CC_SET_TEMPERATURE_REGULATION,&t,NULL);
  afreeze = 1;

  spar.ccd = readout->chip;
  spar.readoutMode = readout->mode;
  spar.top = readout->x;
  spar.left = readout->y;
  spar.height = readout->height;
  spar.width = readout->width;

#ifdef DEBUG
  printf("starting camera readout...\n");
#endif
  if( ! sbu(CC_START_READOUT,&spar,NULL) ) {
    destroy_readout(readout);
    pthread_exit(NULL);
  }
  m = pthread_mutex_lock(&mutex);
  if( m != 0 ) syslog(LOG_ERR,"Mutex lock failed:%s(%d)\n",__FILE__,__LINE__);
  pstatus = 0.0;
  pthread_mutex_unlock(&mutex); 

  lpar.ccd = readout->chip;
  lpar.readoutMode = readout->mode;
  lpar.pixelStart =  spar.left;
  lpar.pixelLength = spar.width;

  for( i = 0; i < readout->height && ! readout_stop; i++ ) {
    
    readout_stop = !sbu(CC_READOUT_LINE,&lpar,readout->data+i*readout->width);

    m = pthread_mutex_lock(&mutex);
    if( m != 0 ) syslog(LOG_ERR,"Mutex lock failed:%s(%d)\n",__FILE__,__LINE__);
    pstatus = 99.0*(i + 0.0)/readout->height;
    pthread_mutex_unlock(&mutex); 
  }
  //syslog(LOG_ERR,"pstatus=%f\n",pstatus);

  epar.ccd = readout->chip;
  sbu(CC_END_READOUT,&epar,NULL);

  misc.fanEnable = TRUE;
  misc.shutterCommand = SC_LEAVE_SHUTTER;
  misc.ledState = LED_BLINK_LOW;

  if( ! sbu(CC_MISCELLANEOUS_CONTROL,&misc,NULL) ) {   /* blink LED slowly */
    destroy_readout(readout);
    pthread_exit(NULL);
  }

#ifdef DEBUG
  printf("Autofreeze...OFF \n");
#endif
  t.regulation = REGULATION_UNFREEZE;
  sbu(CC_SET_TEMPERATURE_REGULATION,&t,NULL);
  afreeze = 0;

  /* SBIG protocol lost time synchronisation, be sure that it is correct */
  /*
  device = ccd_get_address(readout->ccd);
  for( i=0; dev[i]; i++)
    if( strcmp(device,dev[i]) == 0 ) {
      time_synchro();
      break;
    }
  */

  /* pthread_mutex_lock(&mutex);*/
  if ( ! readout_stop ) {

#ifdef DEBUG
  printf("saving new image ...\n");
#endif

  /* all files are created with read-only attribute */
  umask( S_IWUSR | S_IWGRP | S_IWOTH );

  output = ccd_get_fitsname(readout->ccd);
  if( strlen(output) < 1 ){
    if( (output = tempnam(TMP,TMPNIGHT)) == NULL) {
      syslog(LOG_ERR,"Failed to get temporary file name.\n");
      destroy_readout(readout);
      pthread_exit(NULL);
    }
    ccd_set_fitsname(readout->ccd,output);
  }

  /* be sure that old file is removed */
  unlink(output);

    binning = readout->mode + 1;
    ccd_keep_image(readout->ccd,readout->chip, binning,
		   readout->x, readout->y, readout->width, 
		   readout->height, readout->data, output);

    /* the temporary files are owned by nobody:nogroup */
    if( getuid() == 0 && chown(output,65534,65534) )
      syslog(LOG_ERR,"chown: %m\n");

  }
  /* pthread_mutex_unlock(&mutex); */

  // finish downloading
  pstatus = 100.1;

  destroy_readout(readout);
#ifdef DEBUG
  printf("terminating readout thread...\n");
#endif
  pthread_exit(NULL);
}


int read_start(CCD *ccd, int bitpix, int mode, int x1, int y1, int x2, int y2, int chip)
{
  readout_par *readout;
  int nx, ny, i;

  /* check if cammera is on */
  if( ! ccd_get_on(ccd) ) {
    syslog(LOG_ERR,"Camere not prepared for readout.\n");
    return(FAIL);
  }

  /* check if exposure is correctly ended */
  i = exp_stat(ccd,chip);
  if( i == RUN ) {
    syslog(LOG_ERR,"Camere not prepared for readout (exposure not ended).\n");
    return(FAIL);
  }

  /* check if readout is in progress */
  i = read_stat(ccd);
  if( 0.0 < i &&  i < 100.0 ) {
    syslog(LOG_ERR,"Camera readout canceled. Readout in progress.\n");
    return(FAIL);
  }

  nx = x2 - x1 + 1;
  ny = y2 - y1 + 1;

  if( nx <= 0 || ny <= 0 )
    return(FAIL);

  if( ! (readout = create_readout(ccd,chip,mode,x1-1,y1-1,nx,ny)) )
    return(FAIL);

  pstatus = 0.0;
  readout_stop = 0;

#ifdef DEBUG
  printf("detaching readout thread..%p\n",readout);
#endif
  if( pthread_create_detach(&thread_readout, NULL, (void *)readout_func, 
			    (void *) readout) ){
    syslog(LOG_ERR,"Failed to create readout thread.\n");
    destroy_readout(readout);
    return(FAIL);
  }
  return(OK);
}

int read_stop(CCD *ccd)
{
  int m;

  m = pthread_mutex_lock(&mutex);
  if( m != 0 ) syslog(LOG_ERR,"Mutex lock failed:%s(%d)\n",__FILE__,__LINE__);
  readout_stop = 1;
  pthread_mutex_unlock(&mutex);
  return(1);
}

char *read_download(CCD *ccd, char *fitsname)
{
  char *file;
  int m;

  if( ! ccd ) 
    return(NULL);

  file = NULL;
  
  if( fitsname == NULL )
    fitsname = ccd_get_fitsname(ccd);

  if( strlen(fitsname) == 0 ) {
    if( (fitsname = tempnam(TMP,TMPNIGHT)) ) {
      syslog(LOG_ERR,"Temporary filename creation failed.\n");
      return(NULL);
    }
    ccd_set_fitsname(ccd,fitsname);
  }

  m = pthread_mutex_lock(&mutex);
  if( m != 0 ) syslog(LOG_ERR,"Mutex lock failed:%s(%d)\n",__FILE__,__LINE__);
  if( ccd && pstatus >= 100.0 && fitsname && 
      (file = malloc(strlen(fitsname)+strlen("file://")+1)) ) {
    strcpy(file,"file://");
    strcat(file,fitsname);
    pstatus = -1.0;
  }
  pthread_mutex_unlock(&mutex);

  return(file);
}

int read_remove(CCD *ccd)
{
  char *fitsname;

  fitsname = ccd_get_fitsname(ccd);
#ifdef DEBUG
  printf("Removing %s\n",fitsname);
#endif
  if( ccd && fitsname ) {
    if( unlink(fitsname) < 0 )
      syslog(LOG_ERR,"unlink: %m\n");
    ccd_set_fitsname(ccd,NULL);
  }
  return(OK);
}

int ccd_def_fitsname(CCD *ccd, char *name)
{
  return(ccd_set_fitsname(ccd,name));
}

int ccd_def_object(CCD *ccd, char *name)
{
  return(ccd_set_object(ccd,name));
}

int ccd_def_observer(CCD *ccd, char *name)
{
  return(ccd_set_observer(ccd,name));
}

void ccd_get_telescope_ra_dec(CCD *ccd, double *ra, double *dec)
{
  char output[81];
  char *argv[] = { "telescope","get","-ra", "-dec", NULL };

  *ra = -999.9;
  *dec = -999.9;

  if( exec_command(argv[0],argv,output,80) == 0 ) {

#ifdef DEBUG
    fprintf(stderr,">%s<",output);
#endif

    if( sscanf(output,"%lf %lf",ra,dec) != 2 ) {
      *ra  = -999.9;
      *dec = -999.9;
    }
  }

}

