
/*
   astronomical quantities subroutines
   
   $Id: astro.c,v 1.8 2004-10-07 14:15:46 hroch Exp $   

*/


#include <sys/time.h>
#include <stdio.h>
#include <math.h>
#include "astro.h"

#define xDEBUG_ASTRO

#define xTEST_APPARENT

/* 
   Julian date of current time 
*/
double julian(void)
{
  struct timeval t;
  const double jd1970 = 2440587.5L;
  const double milion = 1e6L;
  const double daysec = 86400.0L;
 
  gettimeofday(&t,NULL);
  /* julday = jd(1970-01-01.0) + frac. of day in UT */
  return(jd1970 + (t.tv_sec + t.tv_usec/milion)/daysec);
}

/*
  convert Equatorial coordinates to horizontal
*/
void conve(double jd, double ra, double dec, double latitude, double longitude,
	   double *a, double *z, double *h)
{
  double r, ha;

  /* legend:
     ra, dec ... right ascension and decliantion
     t .. local siderical time
     a,z .. azimut and altitude above horizont
     h .. hour angle
  */

  ha = lmst(jd,longitude)*15.0L - ra;
  *h = per(ha,360.0L);
  eq2hor(ha,dec,latitude,a,z);


  r = refract(*z);
  *z = *z + r;

#ifdef DEBUG_ASTRO
  printf("hour angle: %lf      refraction: %lf\n",ha,r);
#endif

  if( *a > 360.0 )
    *a = *a - 360.0;
  if( *a < 0.0 )
    *a = *a + 360.0;

}

/*
  convert horizontal to Equatorial coordinates
*/
void conve1(double jd, double a, double z, double latitude, double longitude,
	    double *ra, double *dec)
{
  double h, eta, r, cosd, d;
  double dr, dd;

  /* legend:
     ra, dec ... right ascension and decliantion
     t .. local siderical time
     a,z .. azimut and altitude above horizont
  */

  hor2eq(a,z,latitude,&h,&d);
  *ra = lmst(jd,longitude)*15.0L - h;
  *dec = d;

  r = refract(z);

  /* computation of angle eta Smart, page 71, from (A) on page 7,
     we identify b = z, c = 90-dec, a = 90-latitude, than
     
                sin(latitude) - cos(z) * sin(dec)
     cos(eta) = ---------------------------------
                       sin(z) * cos(dec)

     for example, for latitude = 50, dec =0 is 

        sin(latitude)
        ------------ = cos(eta)
        sin(z)

     so for z = 90, eta = 40, for latitude=z is eta=0

  */

  eta = sin(z/RAD)*cos(*dec/RAD);
  if( fabs(eta) > EPSILON )
    eta = (sin(latitude/RAD) - cos(z/RAD)*sin(*dec/RAD))/eta;
  else
    eta = 0.0;  /* pole */

  if( -1.0 <= eta && eta <= 1.0 )
    eta = acos(eta);
  else
    eta = 0.0;

  cosd = cos(*dec/RAD);
  if( fabs(cosd) > EPSILON )
    dr = r*sin(eta)/cosd;
  else
    dr = 0.0;
  
  dd = r*cos(eta);

  /*
  *ra = *ra - dr;
  *dec = *dec - dd;
  */

  if( *ra < 0.0 )
    *ra = *ra + 360.0;
  if( *ra > 360.0 )
    *ra = *ra - 360.0;

#ifdef DEBUG_ASTRO
  printf("eta:%lf   delta alpha: %lf delta delta: %lf\n",eta,dr,dd);
#endif

}


void conve0(double jd, double ha, double dec, double latitude, double longitude, double *a, double *z)
{
  /*  double r;*/

  /* legend:
     ha, dec ... houra angle and declination
     t .. local siderical time
     a,z .. azimut and altitude above horizont
     h .. hour angle
  */

  eq2hor(ha,dec,latitude,a,z);

  /*
  *z = *z + r;
  */

#ifdef DEBUG_ASTRO
  double r = refract(*z);
  printf("hour angle: %lf      refraction: %lf\n",ha,r);
#endif

  if( *a > 360.0 )
    *a = *a - 360.0;
  if( *a < 0.0 )
    *a = *a + 360.0;

}

void conve10(double jd, double a, double z, double latitude, double longitude,
	    double *ha, double *dec)
{
  double h, d;

  /* legend:
     ra, dec ... right ascension and decliantion
     t .. local siderical time
     a,z .. azimut and altitude above horizont
  */

  hor2eq(a,z,latitude,&h,&d);

  if( h > 180.0 )
    h = h - 360.0;

  *ha = h;
  *dec = d;
}

double rekta(double mhc, double ha)
{
  return(per(15.0*mhc - ha,360.0));
}

double houra(double mhc, double ra)
{
  return(per(15.0*mhc - ra,360.0));
}

void toappar(double latitude, double ha, double dec, double *ha1, double *dec1)
{
  double a,h,z;

#ifdef TEST_APPARENT
  *ha1 = ha;
  *dec1 = dec;
#else

  eq2hor(ha,dec,latitude,&a,&h);
#ifdef DEBUG_ASTRO
  printf("toappar: pred refrakci: %10.5f\n",h);
#endif
  z = 90.0 - h;
  z = z - refract(z);
  h = 90.0 - z;
#ifdef DEBUG_ASTRO
  printf("toappar: po refrakci: %10.5f\n",h);
#endif
  hor2eq(a,h,latitude,ha1,dec1);

  if( *ha1 > 180.0 )
    *ha1 = *ha1 - 360.0;

#endif
}

void fromappar(double latitude, double aha, double adec, double *ha, double *dec)
{
  double a,h,z;

#ifdef TEST_APPARENT
  *ha = aha;
  *dec = adec;
#else

  eq2hor(aha,adec,latitude,&a,&h);
#ifdef DEBUG_ASTRO
  printf("fromappar: pred refrakci: %10.5f\n",h);
#endif
  z = 90.0 - h;
  z = z + refract(z);
  h = 90.0 - z;
#ifdef DEBUG_ASTRO
  printf("fromappar: po refrakci: %10.5f\n",h);
#endif
  hor2eq(a,h,latitude,ha,dec);

  if( *ha > 180.0 )
    *ha = *ha - 360.0;

#endif
}
