
/*
   mount-client alias telescope

   is a shell interface to the telescoped daemon

   $Id: mount-client.c,v 1.26 2008-09-11 19:31:01 hroch Exp $

*/

#include "nightview.h"
#include "mount.h"
#include "com.h"
#include "progress.h"
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

/* progress meter delay in seconds */
#define PROGRESS_DELAY 2

extern void show_help(void);
extern void interrupt();
extern void terminate();

TELE *tel;

int main(int argc, char *argv[])
{
  int i, l, get, set, sw, sp, verbose, hclock;
  int prsn = 0;
  char *host = MOUNT_LOCALHOST;
  char *prm;
  float x, y, vra, vdec, vclock;
  double tra1, tdec1, tdome1, tra, tdec, tdome, maxvelra, maxveldec, velra, veldec;

  /* set up interrupt  signal handler */
  signal(SIGINT,interrupt);
  signal(SIGTERM,terminate);

  host = getenv(TELESCOPE_HOST);
  if( host == NULL )
    host = MOUNT_LOCALHOST;

  if( argc == 1 ) {
    show_help();
    return(0);
  }

  get = 0; set = 0;
  verbose = 1;
  vra = -1.0; vdec = -1.0; vclock = -1.0;
  for( i = 1; i < argc; i++) {
    
    /* help */
    if( strcmp(argv[i],"-h") == 0 || strcmp(argv[i],"--help") == 0 ) {
      show_help();
      return(0);
    }

    /* get */
    else if( strcmp(argv[i],"get") == 0 )
      get = i;

    /* set */
    else if( strcmp(argv[i],"set") == 0 )
      set = i;

    /* non-local connection */
    else if( strcmp(argv[i],"-host") == 0 && i+1 < argc )
      host = argv[++i];

    /* verbose */
    else if( strcmp(argv[i],"-q") == 0 )
      verbose = 0;

    /* progress bar on sample line */
    else if( strcmp(argv[i],"-pn") == 0 )
      prsn = 1;
  }

  if( get && set ) {
    fprintf(stderr,"Please, specify only 'get' or 'set', not both.\n");
    return(1);
  }

  if( get == 0 && set == 0) {
    fprintf(stderr,"Please, specify 'get', 'set' or --help.\n");
    return(1);
  }

    /* connect to server */
  if( ! tel_login(host) ) {
    fprintf(stderr,"Connection to %s failed.\n",host);
    return(1);
  }

  if( (tel = tel_init(host)) == NULL ) {
    fprintf(stderr,"Connection to %s failed or telescope is down.\n",host);
    return(1);
  }

  if( get ) {
    
    sw = 0;
    sp = 0;
    for( i = get + 1; i < argc; i++ ) {
      
      /* print RA */
      if( strcmp(argv[i],"-ra") == 0 ) {
	sw = 1;
	printf("%7.3f\n",get_ra(tel));
      }

      else if( strcmp(argv[i],"-dec") == 0 ) {
	sw = 1;
	printf("%7.3f\n",get_dec(tel));
      }
      
      else if( strcmp(argv[i],"-a") == 0 ) {
	sw = 1;
	printf("%7.3f\n",get_azimut(tel));
      }

      else if( strcmp(argv[i],"-z") == 0 ) {
	sw = 1;
	printf("%7.3f\n",get_zenit(tel));
      }
      

      else if( strcmp(argv[i],"-ha") == 0 ) {
	sw = 1;
	printf("%7.3f\n",get_ha(tel));
      }

      else if( strcmp(argv[i],"-jd") == 0 ) {
	sw = 1;
	printf("%13.5f\n",get_jd(tel));
      }

      else if( strcmp(argv[i],"-synchro") == 0 ) {
	sw = 1;
	printf("%d\n",get_synchro(tel));
      }

      else if( strcmp(argv[i],"-status") == 0 ) {
	sw = 1;
	get_status(tel,&hclock,&tra,&tra1,&tdec,&tdec1,&tdome,&tdome1);
	printf("%d %.1f/%.1f %.1f/%.1f %.1f/%.1f\n",hclock,tra,tra1,tdec,tdec1,tdome,tdome1);
      }
      else if( strcmp(argv[i],"-velstat") == 0 ) {
	sw = 1;
	get_velstat(tel,&velra,&veldec,&maxvelra,&maxveldec);
	printf("%.3f/%.3f %.3f/%.3f\n",velra,maxvelra,veldec,maxveldec);
      }

      else if( strcmp(argv[i],"-host") == 0 && i+1 < argc )
	i++;

      else {
	fprintf(stderr,"Unknown option: %s.\n",argv[i]);
	return(1);
      } 
    }  

    if( sw == 0 ) {
      printf("Right Ascension = %7.3f\n",get_ra(tel));
      printf("Declination     = %7.3f\n",get_dec(tel));
      printf("Apparent R.A.   = %7.3f\n",get_ara(tel));
      printf("Apparent Dec    = %7.3f\n",get_adec(tel));
      printf("Azimuth         = %7.3f\n",get_azimut(tel));
      printf("Altitude        = %7.3f\n",get_zenit(tel));
      printf("Hour angle      = %7.3f\n",get_ha(tel));
      printf("Dome            = %7.3f\n",get_dome(tel));
      printf("Julian date     = %13.5f\n",get_jd(tel));
      printf("Siderical time  = %9.4f\n",get_mhc(tel));
      printf("Synchronized    = %d\n",get_synchro(tel));
    }

  } /* get */

  for( i = 1; i < argc; i++ ) {

    /* set velocity of RA */
    if( strcmp(argv[i],"-vra") == 0 && i+1 < argc ) {
      if( sscanf(argv[++i],"%f",&x) == 1 ) {
	vra = x;
      }
      else {
	fprintf(stderr,"Specify velocity as a number.\n");
	return(1);
      }
    }
    
    /* set velocity of Dec */
    if( strcmp(argv[i],"-vdec") == 0 && i+1 < argc ) {
      if( sscanf(argv[++i],"%f",&x) == 1 ) {
	vdec = x;
      }
      else {
	fprintf(stderr,"Specify velocity as a number.\n");
	return(1);
      }
    }

    /* set velocity of clock */
    if( strcmp(argv[i],"-vclock") == 0 && i+1 < argc ) {
      if( sscanf(argv[++i],"%f",&x) == 1 ) {
	vclock = x;
      }
      else {
	fprintf(stderr,"Specify velocity as a number.\n");
	return(1);
      }
    }
  }

  if( set ) {

    sw = 0;
    sp = 0;
    for( i = set + 1; i < argc; i++ ) {
      
      /* set equatorial coordinates */
      if( strcmp(argv[i],"-coo") == 0 && i + 2 < argc ) {
	sw = 1;
	if( sscanf(argv[++i],"%f",&x) == 1 && (  0.0 <= x && x < 360.0) && 
	    sscanf(argv[++i],"%f",&y) == 1 && (-90.0 <= y && y <= 90.0) ) {
	  if( set_coo(tel, x, y, vra, vdec) == FAIL ) {
	    fprintf(stderr,"Equatorial coordinates set failed.\n");
	    return(1);
	  }
	  sp = 1;
	}
	else {
	  fprintf(stderr,"Wrong format or range for a coordinate [degree].\n");
	  return(1);
	}
      }
      if( strcmp(argv[i],"-coa") == 0 && i + 2 < argc ) {
	sw = 1;
	if( sscanf(argv[++i],"%f",&x) == 1 && (  0.0 <= x && x < 360.0) && 
	    sscanf(argv[++i],"%f",&y) == 1 && (-90.0 <= y && y <= 90.0) ) {
	  if( set_coa(tel, x, y, vra, vdec) == FAIL ) {
	    fprintf(stderr,"Equatorial coordinates of I. kind set failed.\n");
	    return(1);
	  }
	  sp = 1;
	}
	else {
	  fprintf(stderr,"Wrong format or range for a coordinate [degree].\n");
	  return(1);
	}
      }


      /* calibrate */
      if( strcmp(argv[i],"-cal") == 0 && i + 2 < argc ) {
	sw = 1;
	if( sscanf(argv[++i],"%f",&x) == 1 && (  0.0 <= x && x < 360.0) &&
	    sscanf(argv[++i],"%f",&y) == 1 && (-90.0 <= y && y <= 90.0) ) {
	  if( set_calibrate(tel, x, y) == FAIL ) {
	    fprintf(stderr,"Calibration failed.\n");
	    return(1);
	  }
	}
	else {
	  fprintf(stderr,"Wrong format or range for coordinate [degree].\n");
	  return(1);
	}
      }

      /* add to RA */
      if( strcmp(argv[i],"+ra") == 0 && i+1 < argc ) {
	sw = 1;
 	prm = argv[++i];
        if( strcmp(prm,"!") == 0 ) {
	  if( stop_ra(tel) == 0 ) {
	    fprintf(stderr,"Stop command for RA failed.\n");
	    return(1);
	  }
	}
	else if( strcmp(prm,"+") == 0 || strcmp(prm,"-") == 0) {
	  if( set_xra(tel,prm,vra) == FAIL ) {
	    fprintf(stderr,"Right Ascension non-stop mode set failed.\n");
	    return(1);
	  }
	  sp = 1;
	}
	else if( sscanf(prm,"%f",&x) == 1 ) {
	  if( set_dra(tel,x,vra) == FAIL ) {
	    fprintf(stderr,"Right Ascension as difference set failed.\n");
	    return(1);
	  } 
	  sp = 1;
	}
	else {
	  fprintf(stderr,"Specify a number or a symbol to R.A. step.\n");
	  return(1);
	}
      }

      /* add to dec */
      if( strcmp(argv[i],"+dec") == 0 && i+1 < argc ) {
	sw = 1;
 	prm = argv[++i];
	if( strcmp(prm,"!") == 0 ) {
	  if( stop_dec(tel) == 0 ) {
	    fprintf(stderr,"Stop command for Dec failed.\n");
	    return(1);
	  }
	}
	else if( strcmp(prm,"+") == 0 || strcmp(prm,"-") == 0 ) {
	  if( set_xdec(tel,prm,vdec) == FAIL ) {
	    fprintf(stderr,"Declination non-stop mode set failed.\n");
	    return(1);
	  } 
	  sp = 1;
	}
	else if( sscanf(prm,"%f",&x) == 1 ) {
	  if( set_ddec(tel,x,vdec) == FAIL ) {
	   fprintf(stderr,"Declination difference set failed.\n");
	    return(1);
	  } 
	  sp = 1;
	}
	else {
	  fprintf(stderr,"Specify a number to Decl. step.\n");
	  return(1);
	}
      }

      /* start clock machine */
      if( strcmp(argv[i],"-c") == 0 && i+1 < argc && strcmp(argv[i+1],"on") == 0 ){
	sw = 1;
	i++;
	if( clock_start(tel,vclock) == 0 )
	  fprintf(stderr,"Clock machine start failed.\n");
      }

      /* stop clock machine */
      if( strcmp(argv[i],"-c") == 0 && i+1 < argc && strcmp(argv[i+1],"off") == 0 ){
	sw = 1;
	i++;
	if( clock_stop(tel) == 0 )
	  fprintf(stderr,"Clock machine stop failed.\n");
      }
      
      /* park telescope */
      if( strcmp(argv[i],"-c") == 0 && i+1 < argc && strcmp(argv[i+1],"park") == 0 ){
	sw = 1;
	i++;
	if( park(tel) == 0 )
	  fprintf(stderr,"Parking failed.\n");
      }
      

    } /* for */

    if( sw == 0 ) {
      fprintf(stderr,"Nothing to set.\n");
      return(1);
    }

  } /* set */

  /* print progress bars */
  if( sp && verbose ) {

    l = fprintf(stdout,"Telescope moving...");
    fflush(stdout);
    sleep(PROGRESS_DELAY);

    while( 1 ) {

      for(i=0; i<l;i++)
	fprintf(stdout,"\b");
      l = 0;

      get_status(tel,&hclock,&tra,&tra1,&tdec,&tdec1,&tdome,&tdome1);
      if( tra > 0.0 && tra1 > 0.0 )
	l += fprintf(stdout," [Right Ascension %.0f/%.1f] ",tra,tra1);

      get_status(tel,&hclock,&tra,&tra1,&tdec,&tdec1,&tdome,&tdome1);
      if( tdec > 0.0 && tdec1 > 0.0 )
	l += fprintf(stdout, " [Declination %.0f/%.1f] ",tdec, tdec1);

      get_status(tel,&hclock,&tra,&tra1,&tdec,&tdec1,&tdome,&tdome1);
      if( tdome > 0.0 && tdome1 > 0.0)
	l += fprintf(stdout," [Dome %.0f/%.1f] ", tdome, tdome1);
      
      fflush(stdout);

      /* nonportable! C's comparing of zeros is nonportable!! */
      if( fabs(tra) < 1e-5 && fabs(tdec) < 1e-5 && fabs(tdome) < 1e-5 )
	break;
      else
	sleep(PROGRESS_DELAY);
    }

    for(i=0; i<l;i++)
      fprintf(stdout,"\b");
    for(i=0; i<l;i++)
      fprintf(stdout,"  ");
    for(i=0; i<l;i++)
      fprintf(stdout,"\b");

  }
  else if( sp && prsn ) {

    fprintf(stdout,"Telescope moving...\n");
    sleep(PROGRESS_DELAY);

    while( 1 ) {

      get_status(tel,&hclock,&tra,&tra1,&tdec,&tdec1,&tdome,&tdome1);
      if( tra > 0.0 && tra1 > 0.0 )
	fprintf(stdout," [Right Ascension %.0f/%.1f] ",tra,tra1);

      get_status(tel,&hclock,&tra,&tra1,&tdec,&tdec1,&tdome,&tdome1);
      if( tdec > 0.0 && tdec1 > 0.0 )
	fprintf(stdout," [Declination %.0f/%.1f] ",tdec, tdec1);

      get_status(tel,&hclock,&tra,&tra1,&tdec,&tdec1,&tdome,&tdome1);
      if( tdome > 0.0 && tdome1 > 0.0)
	fprintf(stdout," [Dome %.0f/%.1f] ",tdome, tdome1);

      fprintf(stdout,"\n");
      fflush(stdout);

      /* nonportable! C's comparing of zeros is nonportable!! */
      if( fabs(tra) < 1e-5 && fabs(tdec) < 1e-5 && fabs(tdome) < 1e-5 )
	break;
      else
	sleep(PROGRESS_DELAY);
	
    }
  }

  return(0);
}


void show_help()
{
  printf("%s\n",COOLVER);
  printf("Telescope mount position control\n");
  printf("Usage: telescope [options] [set | get] [commands]\n");
  printf("options:\n");
  printf("\t get:\t (specify nothing, one or more option)\n");
  printf("\t\t -ra   print Right Ascension in degrees\n");
  printf("\t\t -dec  print Declination in degrees\n");
  printf("\t\t -a    print Azimuth (180 deg = sud)\n");
  printf("\t\t -z    print Altitude\n");
  printf("\t\t -ha   print Hour Angle\n");
  printf("\t\t -jd   print Julian date\n");
  printf("\t\t -synchro   sychronisation,(1=values are probably correct)\n");
  printf("\t\t -status   print status of moving telescope\n");
  printf("\t\t -velstat  RA and Dec velocities in format current/max\n\n");
  printf("\t set:\t (every option needs additional argument(s))\n");
  printf("\t\t -coo RA Dec set the Equatorial coordinates in degrees\n");
  printf("\t\t -coa HA Dec set the Equatorial coordinates (I. kind) in degrees\n");
  printf("\t\t -cal RA Dec calibrate Equatorial the coordinates in degrees\n");
  printf("\t\t +ra [ddd.d..] add value (may be negative) to R.A. in degrees\n");
  printf("\t\t +dec [ddd.d..] add value (may be negative) to Dec in degrees\n");
  printf("\t\t +ra [+|-] for continue slew without endpoint\n");
  printf("\t\t +dec [+|-] for continue slew without endpoint\n");
  printf("\t\t +ra ! stop slew in Right Ascension\n");
  printf("\t\t +dec ! stop slew in Declination\n");
  printf("\t\t -vra   set velocity of Right Ascension in degrees per second\n");
  printf("\t\t -vdec  set velocity of Declination in degress per seconds\n");
  printf("\t\t -vclock set velocity of the hour clock\n");
  printf("\t\t -c on   start the hour clock\n");
  printf("\t\t -c off  stop the hour clock\n");
  /*printf("\t\t -c park park (switch off) telescope\n\n");*/
  printf("\t options:\n");
  printf("\t\t -host address[:port] hostname of server, default to local\n");
  printf("\t\t -q don't print telescope status during setting of telescope\n");
  printf("\t\t -pn   show progress indicator (on new line)\n");
  printf("\t\t -h print this info\n\n");
  printf("Setting of environment variable %s is equivalent to -host option.\n",
	 TELESCOPE_HOST);
  printf("\n");

}

void interrupt(int signal)
{
  fprintf(stderr,"Interupted, but slew is still in progress.\n");
  exit(1);
}

void terminate(int signal)
{
  fprintf(stderr,"Terminating slew in progress\n");
  stop_ra(tel);
  stop_dec(tel);
  exit(1);
}
