
/*

   Unix socket server for mount driver.

   Create socket, listen its and calls commands for operation

   This code is strongly inspired by N. Matthew's and R. Stone's
   Beginning Linux Programming 2nd Edition, Wrong Press Ltd. (1999)
   and Infos to glibc.

   $Id: mount-server.c,v 1.12 2008-01-04 17:48:56 hroch Exp $

*/

#include <stdlib.h>
#include <syslog.h>
#include <signal.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include "nightview.h"
#include "mount.h"

/* we suppose that response will be only LEN bytes long*/
#define LEN 80

/* length of server's incoming queue */
#define MAX_CLIENTS 5

static int sock;

void clean_sock(int sig)
{
  /* stop uncontroled motors */
  mount_shutdown();

  syslog(LOG_NOTICE,"Exiting.\n");
  if( shutdown(sock,2) < 0 )
    syslog(LOG_ERR,"Socket shutdown failed.\n");
  if( unlink(MOUNT_SHOCK) < 0 )
    syslog(LOG_ERR,"Unlink: socket failed.\n");

  signal(sig,SIG_DFL);
}

void terminate_daemon(int sig)
{
  clean_sock(sig);
  exit(0);
}

void abort_daemon(int sig)
{
  clean_sock(sig);
  exit(1);
}

void hup_sock(int sig)
{
  syslog(LOG_NOTICE,"HUP received. Nothing done.\n");
  signal(sig,SIG_DFL);
}


void broken_pipe(int sig)
{
  syslog(LOG_NOTICE,"SIGPIPE %d\n",sig);
}

int main(int argc, char *argv[])
{

  struct stat name_stat;
  struct sockaddr_un name, client;
  int cid;
  char data[LEN], *ansfer = NULL;
  size_t size;
#ifndef DEBUG
  int pid;
#endif

  /* initialize signal handlers */
  signal(SIGTERM,terminate_daemon);
  signal(SIGINT,terminate_daemon);     /* Ctrl-C */
  signal(SIGQUIT,terminate_daemon);    /* Ctrl-\ */
  signal(SIGABRT,abort_daemon);
  signal(SIGSEGV,abort_daemon);
  signal(SIGHUP, hup_sock);
  // signal(SIGPIPE, broken_pipe);
  // signal(SIGCHLD, broken_pipe);

  /* initialize log */
  openlog("telescoped",LOG_PID, LOG_NOTICE);
  syslog(LOG_NOTICE,"Local server starting.\n");

#ifndef DEBUG
  /* close unused files and fork */
  /*
  fclose(stdin);
  fclose(stdout);
  */
  fclose(stderr);
  /* 
     The close of stdin and stdout isn't good idea because both
     reads/writes to childs in move.c.
  */

  if( (pid = fork()) == 0 )
    goto go;
  else if( pid == -1 ) {
    syslog(LOG_ERR,"Daemonization failed: %m\n");
    return(1);
  }
  else
    return(0);

 go:
#endif

  /* initialize commands */
  mount_init(argc, argv);
 
  /* check if server not running */
  if( stat(MOUNT_SHOCK,&name_stat) == 0 ) {
    syslog(LOG_ERR,"Socket descriptor %s exist.\n Remove it manually if server don't runnig and restart me.",MOUNT_SHOCK);
    return(1);
  }

  /* umask for created socket */
  umask(0111);

  /* create server socket */
  if( (sock = socket(PF_LOCAL, SOCK_STREAM, 0)) < 0 ) {
    syslog(LOG_ERR,"socket: %m\n");
    return(1);
  }
  
  /* init server socket */
  name.sun_family = AF_LOCAL;
  strncpy(name.sun_path,MOUNT_SHOCK, sizeof(name.sun_path));
  size = sizeof (name);
  if( bind(sock, (struct sockaddr *) &name, size) < 0) {
    syslog(LOG_ERR,"bind: %m\n");
    return(1);
  }

  /* init incoming call queue */
  if( listen(sock, MAX_CLIENTS) < 0 ) {
    syslog(LOG_ERR,"listen: %m\n");
    return(1);
  }
    
  /* we are waiting for clients */
  while(1) {

    /*  accepting incoming call */
    socklen_t socklen = sizeof(client);
    if( (cid = accept(sock, (struct sockaddr *) &client, &socklen)) < 0 ) {
      syslog(LOG_ERR,"accept: %m\n");
      return(1);
    }
    /*
    syslog(LOG_NOTICE,"Conection on socked %d accepted\n",cid);
    */

    /* data read */
    if( (size = read(cid,data,LEN)) < 0 ) {
      syslog(LOG_ERR,"sread: %m\n");
      return(1);
    }

#ifdef DEBUG
    printf("server recived: %s\n",data);
#endif

    if( size > 0 ) {

      /* transfer data to the control routine */
      data[size] = '\0';
      if( (ansfer = mount(data,&size)) == NULL || size < 0 ) {
	syslog(LOG_ERR,"mount: %m\n");
	return(1);
      }
 
#ifdef DEBUG
      printf("server send: %s\n",ansfer);
#endif

      /* response */
      if( write(cid,ansfer,size) < 0 ) {
	syslog(LOG_ERR,"swrite: %m\n");
	return(1);
      }
    }

    /* close socket to the client */
    if( close(cid) < 0 ) {
      syslog(LOG_ERR,"close: %m\n");
      return(1);
    }
    if( ansfer ) 
      free(ansfer);
    
  } /* end of loop */

  return(0);
}
