 
/*

   main driver
   
   $Id: mount.c,v 1.34 2007-08-29 20:10:53 hroch Exp $
   
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <pthread.h>
#include <unistd.h>
#include <syslog.h>
#include <sys/time.h>
#include <time.h>
#include <math.h>
#include "mount.h"
#include "rcfile.h"
#include "astro.h"
#include "exec.h"
#include "nightview.h"

/* axes */
static AXIS aha, adec, adome;
static CLOCK aclock;
static TELE tel;

/*
static double setpoint_ha, setpoint_dec, setpoint_ra, old_hour;
*/

double old_hour;

static int iclock, iha, idec, idome;
static int stop_main_loop;
static pthread_t th_main;
static pthread_mutex_t mut;
static pthread_cond_t cond;

/* file to save position */
static char *file_pos = NULL;

/* format for current position file read, step over first '=' */
#define FORMAT "%*[^=] %*s %lf"

/* config file, shared with nightview */
/*#define SYSTEMRC "/usr/local/etc/nightviewd.conf"*/

/* function to save of the actual values  */
int data_save(char *file, TELE *t)
{
  FILE *f;

  if( (f = fopen(file,"w")) ) {
    fprintf(f,"JD(UT) = %f\n",t->jd);
    fprintf(f,"RA(deg) = %f\n",t->ra);
    fprintf(f,"dec(deg) = %f\n",t->dec);
    fprintf(f,"dome(deg) = %f\n",t->dome);
    fprintf(f,"latitude(deg) = %f\n",t->latitude);
    fprintf(f,"longitude(deg) = %f\n",t->longitude);
    fprintf(f,"azimut(deg) = %f\n",t->azimut);
    fprintf(f,"zenit distance(deg) = %f\n",t->zenit);
    fprintf(f,"hour angle(deg) = %f\n",t->ha);
    fprintf(f,"mean siderical time (deg) = %f\n",t->mhc);
    fprintf(f,"RAapp(deg) = %f\n",t->ara);
    fprintf(f,"DECapp(deg) = %f\n",t->adec);
    fprintf(f,"HAapp(deg) = %f\n",t->aha);
    fprintf(f,"Spin = %f\n",t->spin);
    fclose(f);
    return(0);
  }
  else
    return(1);
}

/* This function, called as thread, controls of the telescope and dome
   and performs non-invoked user action */

void main_loop(void)
{
  struct timespec t0,t1;
  double azdome, jd_save;
  int ha_stoped, dec_stoped, ra_move, dec_move;

  old_hour = tel.ha;
  jd_save = 0;
  while(stop_main_loop) {
    /* sleep */
    t0.tv_sec = 0;
    t0.tv_nsec = LOOP_STEP;
    nanosleep(&t0,&t1);

    iha = update_axis(&aha,-1.0L);
    idec = update_axis(&adec,-1.0L);
    idome = update_axis(&adome,DOME_INTERVAL/86400.0);

    ra_move = aget_move(&aha) == 1;
    dec_move = aget_move(&adec) == 1;

    ha_stoped = aget_stoped(&aha);
    dec_stoped = aget_stoped(&adec);

    if( ha_stoped && aget_clock(&aclock) )
      aset_clock(&aclock,-1.0);
    iclock = update_axis_clock(&aclock);

    pthread_cond_broadcast(&cond);

    pthread_mutex_trylock(&mut);

    /* update aux. quantities */
    tel.jd = julian();
    tel.mhc = lmst(tel.jd,tel.longitude);
    tel.clock = aget_clock(&aclock);

    /* printf("MHC = %lf\n",tel.mhc);*/

    /* update coordinates */
    if( ha_stoped == 1 || dec_stoped == 1 ) {

      if( ha_stoped == 1 ) {
	tel.aha = aget_coostop(&aha);
	/*
	  tel.ha = setpoint_ha;
	  tel.ra = setpoint_ra;
	*/
	aset_stoped(&aha,0);
      }

      if( dec_stoped == 1 ) {
	tel.adec = aget_coostop(&adec);
	/*
	  tel.dec = setpoint_dec;
	*/
	aset_stoped(&adec,0);
      }
      fromappar(tel.latitude,tel.aha,tel.adec,&tel.ha,&tel.dec);
      tel.ra = rekta(tel.mhc,tel.ha);
    } 

    if( ra_move || dec_move ) {

      tel.aha = aget_coo(&aha);
      tel.adec = aget_coo(&adec);
      tel.ara = rekta(tel.mhc,tel.aha);
      fromappar(tel.latitude,tel.aha,tel.adec,&tel.ha,&tel.dec);
      tel.ra = rekta(tel.mhc,tel.ha);
    }
    else {

      if( tel.clock == 1 ) {
	tel.ha = houra(tel.mhc,tel.ra);
	toappar(tel.latitude,tel.ha,tel.dec,&tel.aha,&tel.adec);
	aset_cal(&aha,tel.aha);
	aset_cal(&adec,tel.adec);
	tel.ara = rekta(tel.mhc,tel.aha);
      }
      else if( tel.clock == 0 ){
	conve10(tel.jd,tel.azimut,tel.zenit,tel.latitude,tel.longitude,
		&tel.aha,&tel.adec);
	tel.ara = rekta(tel.mhc,tel.aha);
	fromappar(tel.latitude,tel.aha,tel.adec,&tel.ha,&tel.dec);
	tel.ra = rekta(tel.mhc,tel.ha);
	aset_cal(&aha,tel.aha);
	aset_cal(&adec,tel.adec);
      }

    }
    conve0(tel.jd,tel.aha,tel.adec,tel.latitude,tel.longitude,&tel.azimut,&tel.zenit);

    /* keep interval info for hour angle */
    tel.spin = tel.spin + izero(old_hour,tel.ha);
    old_hour = tel.ha;
    /*
    if( abs(tel.spin) > 0.1 )
      printf("spin = %lf\n",tel.spin);
    */

    /* synchro dome */
    tel.dome = aget_coo(&adome);
    azdome = diff_ra(tel.dome,tel.azimut);
    if( azdome > DOME_STEP )
      aset_coo(&adome,azdome,0);

    /* save position */
    if( jd_save < tel.jd ) {
      jd_save = tel.jd + COMMIT_DELAY/86400.0;
      if( data_save(file_pos,&tel) != 0 )
	syslog(LOG_ERR,"Saving of the current position failed: %m\n");
    }

    /* sanity check */
    if( tel.zenit < 0.0 ) {
      syslog(LOG_CRIT,"Slew under horizont: h= %lf\n",tel.zenit);
      tel.synchro = 0;
      aset_stop(&aha);
      aset_stop(&adec);
      tel.zenit = 1e-5;
      conve10(tel.jd,tel.azimut,tel.zenit,tel.latitude,tel.longitude,
	      &tel.aha,&tel.adec);
      tel.ara = rekta(tel.mhc,tel.aha);
      fromappar(tel.latitude,tel.aha,tel.adec,&tel.ha,&tel.dec);
      tel.ra = rekta(tel.mhc,tel.ha);
      aset_cal(&aha,tel.aha);
      aset_cal(&adec,tel.adec);
      //      abort();
    }
    pthread_mutex_unlock(&mut);

  }
  pthread_exit(NULL);
}

int mount_init(int argc, char *argv[])
{
  FILE *f, *rc;
  int i;
  char *val, line[BLOCK];
  double vel;
  char *conf = NULL;
 
  iclock = 0;
  iha = 0;
  idec = 0;
  idome = 0;

  /* init position to unknown values */
  init_axis(&aha);
  init_axis(&adec);
  init_axis(&adome);
  init_clock(&aclock);
  set_axis_start_command(&aha,REKL_COMMAND);
  set_axis_stop_command(&aha,REKL_COMMAND);
  set_axis_clock_command(&aclock,CLOCK_COMMAND);
  set_axis_start_command(&adec,DEKL_COMMAND);
  set_axis_stop_command(&adec,DEKL_COMMAND);
  /*
  set_axis_start_command(&adome,DOME_COMMAND);
  set_axis_stop_command(&adome,DOME_COMMAND);
  */
  set_axis_start_command(&adome,NULL);
  set_axis_stop_command(&adome,NULL);

  tel.jd = 0.0;
  tel.ra = 0.0;
  tel.dec = 0.0;
  tel.azimut = 0.0;
  tel.zenit = 0.0;
  tel.ha = 0.0;
  tel.dome = 0.0;
  tel.clock = 0;
  tel.mhc = 0.0;
  tel.longitude = 0.0;
  tel.latitude = 0.0;
  tel.spin = 0.0;
  tel.synchro = 0;

  /* init position to saved values, if a position_file exist */
  if( file_pos == NULL )
    file_pos = POSITION_FILE;
  
  if( (f = fopen(file_pos,"r")) ) {
    fscanf(f,FORMAT,&tel.jd);
    fscanf(f,FORMAT,&tel.ra);
    fscanf(f,FORMAT,&tel.dec);
    fscanf(f,FORMAT,&tel.dome);
    fscanf(f,FORMAT,&tel.latitude);
    fscanf(f,FORMAT,&tel.longitude);
    fscanf(f,FORMAT,&tel.azimut);
    fscanf(f,FORMAT,&tel.zenit);
    fscanf(f,FORMAT,&tel.ha);
    fscanf(f,FORMAT,&tel.mhc);
    fscanf(f,FORMAT,&tel.ara);
    fscanf(f,FORMAT,&tel.adec);
    fscanf(f,FORMAT,&tel.aha);
    fscanf(f,FORMAT,&tel.spin);
    fclose(f);
    aset_cal(&aha,tel.aha);
    aset_cal(&adec,tel.adec);
    tel.ara = houra(tel.mhc,tel.aha);
    fromappar(tel.latitude,tel.aha,tel.adec,&tel.ha,&tel.dec);
    tel.ra = houra(tel.mhc,tel.ha);
    tel.spin = 0.0;

#ifdef DEBUG
    printf("Coord init: %f %f %f %f %f %f %f\n",tel.jd,tel.ra,tel.dec,tel.azimut,tel.zenit,tel.ha,tel.dome);
#endif

    /* test on tel.zenit > 0 */
    if( tel.zenit < 0.0 ) {
      tel.zenit = 0.0;
      tel.azimut = 0.0;
      tel.ha = 0.0;
      tel.dec = tel.latitude;
    }
  }
  else
    syslog(LOG_ERR,"The telescope initialisation failed. Calibration required.\n");


  /* check for config file */
  for( i = 1; i < argc; i++) {
    if( strcmp(argv[i],"-c") == 0 && i++ < argc )
      conf = argv[i];
  }

  /* init by config file - read the rc file */
  if( conf == NULL ) conf = SYSTEMRC;
  if( (rc = rc_open(conf,"r")) ) {
    while( rc_gets(line,BLOCK,rc) ) {
      if( (val = strchr(line,'=')) && strstr(line,"Longitude") ) {
	if( sscanf(val+1,"%lf",&tel.longitude) != 1 )
	  tel.longitude = 0.0;
      }
      if( (val = strchr(line,'=')) && strstr(line,"Latitude") ) {
	if( sscanf(val+1,"%lf",&tel.latitude) != 1 )
	  tel.latitude = 0.0;
      }
    }
    rc_close(rc);
  }

  /* init by program arguments */
  for( i = 1; i < argc; i++) {
    if( strcmp(argv[i],"-longitude") == 0 && i++ < argc )
      if( sscanf(argv[i],"%lf",&tel.longitude) != 1 )
	tel.longitude = 0.0;
    if( strcmp(argv[i],"-latitude") == 0 && i++ < argc )
      if( sscanf(argv[i],"%lf",&tel.latitude) != 1 )
	tel.latitude = 0.0;
  }


#ifdef DEBUG
  printf("Observatory coordinates: %f %f\n",tel.latitude, tel.longitude);
#endif

  /* set up velocities */
  if( initvelo(REKL_COMMAND,"-velmax",&vel) )
    aset_velmax(&aha,vel);
  else  
    syslog(LOG_ERR,"'%s -velmax' failed. Certain functions are not available.\n",REKL_COMMAND);

  if( initvelo(REKL_COMMAND,"-veldef",&vel) )
    aset_veldef(&aha,vel);  
  else      
    syslog(LOG_ERR,"'%s -veldef' failed. Certain functions are not available.\n",REKL_COMMAND);

  if( initvelo(DEKL_COMMAND,"-velmax",&vel) )
    aset_velmax(&adec,vel);
  else
    syslog(LOG_ERR,"'%s -velmax' failed. Certain functions are not available.\n",DEKL_COMMAND);

  if( initvelo(DEKL_COMMAND,"-veldef",&vel) )
    aset_veldef(&adec,vel);
  else
    syslog(LOG_ERR,"'%s -veldef' failed. Certain functions are not available.\n",DEKL_COMMAND);

  /* start the control loop */
  stop_main_loop = 1;
  pthread_mutex_init(&mut,NULL);
  pthread_cond_init(&cond,NULL);
  if( pthread_create(&th_main, NULL, (void *) main_loop, NULL) != 0 ) {
    syslog(LOG_ERR,"Starting main loop failed. An apocalypse is on the way....\n");
  }

  return(0);
}


char *mount(char *in, size_t *size)
{
  char *o,*i;
  char *args[3];
  char direct[2];
  double az, ha, x, y, vx, vy;
  double ra, dec, dra, ddec, ddome, aaha, aadec;
  int j, dir;

#ifdef DEBUG_EXTRA
  printf("mount: %s\n",i);
#endif

  *size = 0;
  if( (o = malloc(BLOCK)) == NULL )
    return(NULL);

  /* Non-execution part of the driver. The variables there are
     only initialized and so on. The commands are executed in 
     next part.
  */

  /* --- login to server --- */
  if( cmpkey(LOGIN,in) )
    snprintf(o,BLOCK,"%s",WELCOME);


  /* --- report status of motors motions --- */
  else if( cmpkey(TEL_STATUS,in) ) {
    snprintf(o,BLOCK,"%f %f %f %f",aget_vel(&aha),aget_vel(&adec),
	     aget_velmax(&aha),aget_velmax(&adec));
  }


  /* --- report telescope information --- */
  else if( cmpkey(TEL_INFO,in) ) {
    
    snprintf(o,BLOCK,"%f %f %f %f %f %f %f %f %f %f %d",tel.jd,tel.ra,tel.ara,
	     tel.dec,tel.adec,tel.dome,tel.azimut,tel.zenit,tel.ha,tel.mhc,
	     tel.synchro);

  }


  /*  --- calibration --- */
  else if( (i = cmpkey(TEL_CALIBRATE,in)) ) {
    
    if( sscanf(i,"%lf %lf",&tel.ra, &tel.dec) == 2 ) {
      tel.ha = rekta(tel.mhc,tel.ra);
      toappar(tel.latitude,tel.ha,tel.dec,&tel.aha,&tel.adec);
      conve0(tel.jd, tel.aha, tel.adec, tel.latitude, tel.longitude,&tel.azimut, &tel.zenit);
      tel.ara = houra(tel.mhc,tel.aha);
      tel.spin = 0.0;
      old_hour = tel.aha;
      aset_cal(&aha,tel.aha);
      aset_cal(&adec,tel.adec);
      if( aget_clock(&aclock) )
	tel.synchro = 1;
      pthread_cond_wait(&cond,&mut);
      strcpy(o,TEL_CALIBRATED);
    }
    else
      strcpy(o,TEL_CALIBRATEDFAIL);
  }

  /*
    This is a part of driver runs external commands.
    We supposing  all these commands as a long timed.
    Some timer is lunched after starting every command (operation).
    A status function (command) is associated with every
    timer. The user can get a status of the timer (operation)
    or it can be interrupted.
  */

  /* --- set telescope to equatorial coordinates --- */
  else if( (i = cmpkey(TEL_SET,in))) {
    
    /* the coordinate setpoint */
    if( sscanf(i,"%lf %lf %lf %lf",&x,&y,&vx,&vy) == 4 ) {
      ra = tel.ra;
      dec = tel.dec;
      if( 0.0 <= x && x < 360.0 ) ra = x;
      if( -90.0 <= y && y <= 90.0 ) dec = y;
      if( vx < 0.0 ) vx = aget_velmax(&aha);
      if( vy < 0.0 ) vy = aget_velmax(&adec);

      /* estimate final coordinates */
      ha = houra(tel.mhc,ra);
      toappar(tel.latitude,ha,dec,&aaha,&aadec);
      dra = diff_ha(aaha, tel.aha,tel.spin);
      ddec = diff_dec(aadec, tel.adec);
      if( dra < -999.0 ) {
	syslog(LOG_ERR,"Unexpected values for hour angle: h1=%lf h2=%f\n",aaha,tel.aha);
	//abort();
      }

      /* time for slew */
      /*
      double ta, td;
      ta = dra/vx;
      td = ddec/vy;
      */
      //double ddt = sqrt(ta*ta + td*td);

      /* correction of hour-angle for Earth's rotation*/
      double ddt = fabs(dra)/vx;
      double dmhc = ddt/3600.0L; /*360.0L/86164.25L*ddt;*/
      
      ha = houra(tel.mhc+dmhc,ra);
      toappar(tel.latitude,ha,dec,&aaha,&aadec);
      
      dra = diff_ha(aaha, tel.aha,tel.spin);
      ddec = diff_dec(aadec, tel.adec);
      if( dra < -999.0 ) {
	syslog(LOG_ERR,"Unexpected vales for hour angle: h1=%lf h2=%f\n",aaha,tel.aha);
	//abort();
      }

      /* correction of hour-angle for Earth rotation*/
      //dra = dra + 360.0L/86164.25L*ddt;

      /*
      ha = houra(tel.mhc,ra);
      */    

      /*
      setpoint_ra = ra;
      setpoint_dec = dec;
      setpoint_ha = ha;
      */

      /* coordinate differences for apparent coordinates*/
      /*
      toappar(tel.latitude,ha,dec,&aaha,&aadec);
      dra = diff_ha(aaha, tel.aha,vx,tel.spin);
      if( dra < -999.0 ) {
	syslog(LOG_ERR,"Unexpected vales for hour angle: h1=%lf h2=%f\n",aaha,tel.aha);
	abort();
      }
      ddec = diff_dec(aadec, tel.adec);
      */

#ifdef DEBUG
      printf("ra,dec,vel_ra,cel_dec,dra,ddec: %f %f %f %f %f %f\n",tel.ra,tel.dec,vx, vy, dra, ddec);
#endif

#ifdef DEBUG
      printf("--------- Coord set: %f %f %f %f\n",aaha,aadec,dra, ddec);
#endif

    /* switch up all motors */
#ifdef DEBUG
      printf("rekl: %f %f %f %f\n",dra,vx,ha,tel.ha);
      printf("dekl: %f %f %f %f\n",ddec,vy,dec,tel.dec);
      printf("dome: %f %f %f %f\n",ddome, 1.0,az,tel.dome);
#endif

      aset_vel(&aha,vx);
      aset_vel(&adec,vy);
      aset_coo(&aha,dra,0);
      aset_coo(&adec,ddec,0);
      pthread_cond_wait(&cond,&mut);
    }

    if( iha == 1 && idec == 1 )
      strcpy(o,TEL_SETOK);
    else
      strcpy(o,TEL_SETFAIL);

  }

  /* --- set telescope to ha-dec coordinates --- */
  else if( (i = cmpkey(TEL_SET_I,in))) {

    /* the coordinate setpoint */
    if( sscanf(i,"%lf %lf %lf %lf",&x,&y,&vx,&vy) == 4 ) {
      ha = tel.ha;
      dec = tel.dec;
      if( 0.0 <= x && x < 360.0 ) ha = x;
      if( -90.0 <= y && y <= 90.0 ) dec = y;
      if( vx < 0.0 ) vx = aget_velmax(&aha);
      if( vy < 0.0 ) vy = aget_velmax(&adec);

      /* estimate final coordinates */
      toappar(tel.latitude,ha,dec,&aaha,&aadec);
      dra = diff_ha(aaha, tel.aha,tel.spin);
      ddec = diff_dec(aadec, tel.adec);
      if( dra < -999.0 ) {
	syslog(LOG_ERR,"Unexpected vales for hour angle: h1=%lf h2=%f\n",aaha,tel.aha);
	abort();
      }
      
      aset_vel(&aha,vx);
      aset_vel(&adec,vy);
      aset_coo(&aha,dra,0);
      aset_coo(&adec,ddec,0);
      pthread_cond_wait(&cond,&mut);
    }

    if( iha == 1 && idec == 1 )
      strcpy(o,TEL_SETOK);
    else
      strcpy(o,TEL_SETFAIL);

  }

  /* --- set telescope to a difference of coordinates --- */
  else if( (i = cmpkey(TEL_SET_DRA,in)) ) {

    /* the coordinate setpoint */
    j = sscanf(i,"%lf %lf",&dra,&x);
    if( j == 2 ) vx = x;
    if( j >= 1 ) {
      dra = - dra;
      aset_vel(&aha,vx);
      aset_coo(&aha,dra,0);
      pthread_cond_wait(&cond,&mut);
    }

    if( j >= 1 && iha )
      strcpy(o,TEL_SETOK);
    else
      strcpy(o,TEL_SETFAIL);

  }
  /* --- set telescope to a difference of coordinates --- */
  else if( (i = cmpkey(TEL_SET_DDEC,in)) ) {

    /* the coordinate setpoint */
    j = sscanf(i,"%lf %lf",&ddec,&x);
    if( j == 2 ) vy = x;
    if( j >= 1 ) {
      aset_vel(&adec,vy);
      aset_coo(&adec,ddec,0);
      /* ddome? */
      pthread_cond_wait(&cond,&mut);
    }
     
    if( j >= 1 && idec )
      strcpy(o,TEL_SETOK);
    else
      strcpy(o,TEL_SETFAIL);

  }

  /* --- set telescope to continue move --- */
  else if( (i = cmpkey(TEL_SET_XRA,in)) ) {

    /* the coordinate setpoint */
    j = sscanf(i,"%s %lf",direct,&x);
    if( j == 2 ) vx = x;
    if( j >= 1 ) {
      if( strcmp(direct,"+") == 0 )
	dir = 1;
      else
	dir = -1;
      aset_vel(&aha,vx);
      aset_move(&aha,dir);
      pthread_cond_wait(&cond,&mut);
    }

    tel.synchro = 0;
    if( j >= 1 && iha )
      strcpy(o,TEL_SETOK);
    else
      strcpy(o,TEL_SETFAIL);

  }

  else if( (i = cmpkey(TEL_SET_XDEC,in)) ) {

    /* the coordinate setpoint */
    j = sscanf(i,"%s %lf",direct,&x);
    if( j == 2 ) vy = x;
    if( j >= 1 ) {
      if( strcmp(direct,"+") == 0 )
	dir = 1;
      else
	dir = -1;
      aset_vel(&adec,vy);
      aset_move(&adec,dir);
      pthread_cond_wait(&cond,&mut);
    }
     
    tel.synchro = 0;
    if( j >= 1 && idec )
      strcpy(o,TEL_SETOK);
    else
      strcpy(o,TEL_SETFAIL);

  }

  /* --- report telescope moving status --- */
  else if( cmpkey(TEL_GET,in) ) {

    sprintf(o,"%d %f %f %f %f %f %f", aget_clock(&aclock),
            aget_timer(&aha),  aget_duration(&aha),
	    aget_timer(&adec), aget_duration(&adec),
	    aget_timer(&adome),aget_duration(&adome));
  }


  /* --- park telescope --- */
  else if( cmpkey(TEL_PARK,in) ) {

    /* deprecated command, return fail only */

    /* coordinate differences */
    /*
    vx = aget_velmax(&aha);
    vy = aget_velmax(&adec); 
    dra = diff_ha(0.0, tel.ha,vx,tel.spin);
    if( dra < -999.0 ) {
      syslog(LOG_ERR,"Unexpected values for hour angle: h1=%lf h2=%f\n",0.0,tel.ha);
      abort();
    }
    ddec = diff_dec(tel.latitude, tel.dec);

    aset_vel(&aha,vx);
    aset_vel(&adec,vy);
    aset_coo(&aha,dra,0);
    aset_coo(&adec,ddec,0);
    pthread_cond_wait(&cond,&mut);

    if( iha == 1 && idec == 1  )
      snprintf(o,BLOCK,"%s %f %f %f",TEL_PARKING,
	       aget_timer(&aha),aget_timer(&adec),aget_timer(&adome));
    else
      strcpy(o,TEL_SETFAIL);
    */
    strcpy(o,TEL_SETFAIL);

  }

  /* --- clock up --- */
  else if( (i = cmpkey(TEL_CLOCK_ON,in)) ) {

    j = sscanf(i,"%lf %lf",&x,&vx);
    if( j == 1 ) vx = -1.0;
    if( j >= 1 ) {
      aset_velclock(&aclock,vx);
      aset_clock(&aclock,x);
      pthread_cond_wait(&cond,&mut);
    }    

    if( j >= 1 && iclock == 1 ) {
      strcpy(o,TEL_CLOCK_ONOK);
    }
    else
      strcpy(o,TEL_SETFAIL);
  }

  /* --- clock down --- */
  else if( cmpkey(TEL_CLOCK_OFF,in) ) {

    aset_clock_stop(&aclock);
    pthread_cond_wait(&cond,&mut);

    tel.synchro = 0;

    if( iclock == 1 ) {
      strcpy(o,TEL_CLOCK_OFFOK);
    }
    else
      strcpy(o,TEL_SETFAIL);
  }

  /* --- telescope move stop --- */
  /*
  else if( cmpkey(TEL_STOP,in) ) {

    aset_stop(&aha);
    aset_stop(&adec);
    pthread_cond_wait(&cond,&mut);

    if( iha == 1 && idec == 1 ) {
      strcpy(o,TEL_STOPOK);
    }
    else
      strcpy(o,TEL_STOPFAIL);
  }
  */
  /* --- telescope move stop RA --- */
  else if( cmpkey(TEL_STOP_RA,in) ) {

    aset_stop(&aha);
    tel.synchro = 0;
    pthread_cond_wait(&cond,&mut);

    if( iha == 1 ) {
      strcpy(o,TEL_STOPOK);
    }
    else
      strcpy(o,TEL_STOPFAIL);
  }

  /* --- telescope move stop Dec --- */
  else if( cmpkey(TEL_STOP_DEC,in) ) {

    aset_stop(&adec);
    tel.synchro = 0;
    pthread_cond_wait(&cond,&mut);
    
     if( idec == 1 ) {
        strcpy(o,TEL_STOPOK);
     }
     else
        strcpy(o,TEL_STOPFAIL);
  }

  /* --- dome open --- */
  else if( cmpkey(DOME_OPEN,in) ) {

    args[0] = "1";
    args[1] = NULL;
    if( exec_command(DOME_INIT,args,NULL,0) == 0 ) /* openning is coded as 1 */
      strcpy(o,TEL_NA);
    else
      strcpy(o,TEL_NA);
  }


  /* --- dome close --- */
  else if( cmpkey(DOME_CLOSE,in) ) {

    args[0] = "0";
    args[1] = NULL;
    if( exec_command(DOME_INIT,args,NULL,0) == 0 ) /* closing is coded as 0.0 */
      strcpy(o,TEL_NA);
    else
      strcpy(o,TEL_NA);
  }


  /* --- dome set --- */
  /* not implemented yet
  else if( strcmp(DOME_SET,i) == 0) {

    dra = diff_ra(ra, pos_ra);
    if( slew_start_d(DOME_COMMAND,dra,0.0,&td) == 0 )
      strcpy(o,TEL_NA);
    else
      strcpy(o,TEL_NA);
  }
  */
  else
    snprintf(o,BLOCK,"Unrecognized command.");

  *size = strlen(o) + 1;
  return(o);
}

void mount_shutdown(void)
{
  aset_stop(&aha);
  aset_stop(&adec);
  aset_stop(&adome);
  aset_clock_stop(&aclock);
  stop_main_loop = 0;
  pthread_join(th_main,NULL);
  data_save(file_pos,&tel);
}
