
/*

   transformation between astronomical coordinates

   Inspired by SLA lib by P.T.Wallace and Numerical Algorithms 
   by Z.Pokorny. Constants according to Astronomical Almanac.

    $Id: traco.c,v 1.4 2007-07-21 16:32:55 hroch Exp $

*/

#include <math.h>
#include "astro.h"

double per(double x, double period)
{
  double t;
  
  t = fmod(x,period);
  if( t < 0.0L ) 
    t = t + period;
  return(t);
}



double gmst(double jd)
{

  /*

  Greenwich siderical time

  jd is a full Julian date

  The precision is better than 1 second. 
  According to Astronomical Almanac 2000.

  */

  double tu, t;

  tu = (jd - 2451545.0L)/36525.0L;
  t = 24110.54841L + tu*(8640184.812866L + tu*(0.093104L - 6.2e-6L*tu));
  t = t/3600.0L + 24.0L*(jd - floor(jd)) + 12.0L;
  t = per(t,24.0L);
  return(t);
}


double equinox(double jd)
{
  return(0.0L);
}


double lmst(double jd, double longitude)
{

  /*
    
  local siderical time

  jd is a full Julian date
  lamda is a longitude in degrees, -west... +east

  The precision is better than 1 second. 
  According to Astronomical Almanac 2000.

  */


  double t;

  t = gmst(jd) + longitude/15.0L + equinox(jd);
  return(t);

}

void eq2hor(double ha, double dec, double latitude, double *az, double *elev)
{

  /*

  equatorial to horizontal coordinates

  all arguments in degrees

  */

  double sinh, cosh, sind, cosd, sinl, cosl, x,y,z,r;

  sinh = sin(ha/RAD);
  cosh = cos(ha/RAD);
  sind = sin(dec/RAD);
  cosd = cos(dec/RAD);
  sinl = sin(latitude/RAD);
  cosl = cos(latitude/RAD);

  x = -cosh*cosd*sinl + sind*cosl;
  y = -sinh*cosd;
  z = cosh*cosd*cosl + sind*sinl;

  r = sqrt(x*x + y*y);
  if( fabs(r) < EPSILON ) 
    *az = 0.0L;
  else
    *az = RAD*atan2(y,x);
  if( *az < 0.0L ) *az = *az + 360.0L;
  *elev = RAD*atan2(z,r);

}

void hor2eq(double az, double elev, double latitude, double *ha, double *dec)
{

  /*

  horizontal to equatorial coordinates

  all arguments in degrees

  */

  double sina, cosa, sine, cose, sinl, cosl, x, y, z, r;

  sina = sin(az/RAD);
  cosa = cos(az/RAD);
  sine = sin(elev/RAD);
  cose = cos(elev/RAD);
  sinl = sin(latitude/RAD);
  cosl = cos(latitude/RAD);

  x = -cosa*cose*sinl + sine*cosl;
  y = -sina*cose;
  z = cosa*cose*cosl + sine*sinl;

  r = sqrt(x*x + y*y);
  if( fabs(r) < EPSILON )
    *ha = 0.0L;
  else
    *ha = RAD*atan2(y,x);
  *dec = RAD*atan2(z,r);
}


double refract(double z)
{

  /*

    compute refraction angle in degrees 

    Smart: Textbook on sphreical astronomy

    constants for pressure 760mmHg, 10deg C with 
    suffucient accuracy for z < 75 deg 

  */

  double r,tanz;

  if( 0 <= z && z <= 90.0 ) {

    tanz = tan(z/RAD);
    r = 58.16*tanz - 0.067*tanz*tanz*tanz; 
    return(r/3600.0);

  }
  else
    return(0.0);
}

