
/*

  night_batch -  to do serie of exposures
                 replacement for night_control

  $Id: night_batch.cpp,v 1.12 2009-02-09 17:57:08 hroch Exp $

*/

#include "nightview.h"
#include <stdexcept>
#include <cmath>
#include <iostream>
#include <sstream>
#include <vector>
#include <string>
#include <cstring>
#include <cstdlib>
#include <cstdio>
#include <unistd.h>

using namespace std;

static bool debug = false;

enum { RUN_CONTROL, RUN_FLATS, RUN_DARKS };

void show_help(const string& ,const string&, int, const string&, const string&);
string popen2(const string&);
void system2(const string&);
unsigned defn0(const string&);

class Exp_basic {

public:
  Exp_basic(): n(1) { }
  Exp_basic(const string&, const string&, const string&); 
  virtual ~Exp_basic() {}

  virtual void set_format(unsigned);
  virtual void set_format(const char *);

  virtual string name(unsigned) const;
  virtual void print_status(const string&);
  virtual void select_filter(const string&) {}
  virtual void exposure(const string&, const string&);

  string exp_time() const;

protected:
  string exptime, prefix, shutter, format;
  unsigned n;

};

class Exp_light: public Exp_basic {

public:
  Exp_light() { cout << n << endl;}
  Exp_light(const string&, const string&, const string&, const string&);

  virtual void set_format(unsigned);

  virtual string name(unsigned) const;
  virtual void print_status(const string&);
  virtual void select_filter(const string&);
  //virtual void exposure(const string&, const string&);

private:
  string filter;
};

Exp_basic::Exp_basic(const string& p, const string& s, const string& e) {
  prefix=p; 
  shutter = s; 
  exptime = e;
  // setup start number
  n = defn0("ls "+prefix+e+"_*.fits 2>/dev/null|wc -l");
}

Exp_light::Exp_light(const string& p, const string& s, const string& e,
		     const string& f) {
  prefix=p; 
  shutter = s;
  exptime = e;
  filter = f;
  // setup start number
  n = defn0("ls "+prefix+"_*"+f+".fits 2>/dev/null|wc -l");
}

void Exp_basic::set_format(unsigned number) {
  //int decimals = int(log(number)/2.3025851) + 1;
  format = prefix + "%s_%d.fits";
}

void Exp_light::set_format(unsigned number) {
  int order = int(log(number)/2.3025851) + 1;
  char decimals = '1';
  if( 0 < order && order <= 9 )
    decimals = char(order + 48);
  format = prefix + "_%0" + decimals + "d%s.fits";
}

void Exp_basic::set_format(const char *oname) {
  format = oname;
}

string Exp_basic::name(unsigned i) const {
  const int NL = 80;
  char line[NL];
  snprintf(line,NL,format.c_str(),exptime.c_str(),n+i);
  string name(line);
  return(name);
}

string Exp_light::name(unsigned i) const {
  const int NL = 80;
  char line[NL];
  snprintf(line,NL,format.c_str(),n+i,filter.c_str());
  string name(line);
  return(name);
}


void Exp_basic::print_status(const string& name) {
  cout << "Exposure = " << exptime << "sec, name = " << name << ", Status: ";
  cout.flush();
}

void Exp_light::print_status(const string& name) {
  cout << "Exposure = " << exptime << " sec, Filter: " << filter
       << ", name=" << name << ", Status: ";
  cout.flush();
}

void Exp_light::select_filter(const string& par) {
  if( filter != "" ) {
    string arg = "night_filter -p -f "+filter+" "+par+" >& /dev/null";
    system2(arg);
  }
}

void Exp_basic::exposure(const string& name, const string& par)
{
  string arg = "night_exposure -p -t "+exptime+" -o "+name+" "+shutter+par;
  system2(arg);
}

string Exp_basic::exp_time() const
{
  return(exptime);
}


class Exp_exp {

public:
  Exp_exp(): w(0) {}
  Exp_exp(const string& p, const string& s, const string& e)
  { w = new Exp_basic(p,s,e); }
  Exp_exp(const string& p, const string& s, const string& e, const string& f)
  { w = new Exp_light(p,s,e,f);}

  void set_format(unsigned n);
  void set_format(const char *);

  string name(unsigned) const;
  void print_status(const string&);
  void select_filter(const string&);
  void exposure(const string&, const string&);

  string exp_time() const;
 private:
  Exp_basic* w;
};

void Exp_exp::set_format(unsigned n) {

  if( w ) 
    w->set_format(n);
  else
    throw runtime_error("A object not initialized yet.");
}

void Exp_exp::set_format(const char *oname) {

  if( w ) 
    w->set_format(oname);
  else
    throw runtime_error("A object not initialized yet.");
}

string Exp_exp::name(unsigned i) const {

  if( w )
    return(w->name(i));
  else
    throw runtime_error("A object not initialized yet.");
}

void Exp_exp::print_status(const string& par) {

  if( w )
    return(w->print_status(par));
  else
    throw runtime_error("A object not initialized yet.");
}

void Exp_exp::select_filter(const string& par) {

  if( w )
    return(w->select_filter(par));
  else
    throw runtime_error("A object not initialized yet.");
}

void Exp_exp::exposure(const string& name,const string& par) {

  if( w )
    return(w->exposure(name,par));
  else
    throw runtime_error("A object not initialized yet.");
}

string Exp_exp::exp_time() const {

  if( w )
    return(w->exp_time());
  else
    throw runtime_error("A object not initialized yet.");
}


string make_dark(string,string);
void dark_subtract(string, string);
void pointing(string, string, string, string, string);

int main(int argc, char *argv[])
{
  char *oname = NULL;
  string prefix = "image";
  int number = 1;
  int dark = 0;
  int wait = 0;
  string bin = " -b 3";
  string shutter = " -s on";
  string point;
  string point_rad = "30";
  string point_tol = "10";
  string point_coo;
  string chip;
  string region;
  string object;
  string observer;
  string host;
  int run = RUN_CONTROL;
  bool nightview = false;
  vector<Exp_exp> exp;
  string name, dname;

  if( strstr(argv[0],"night_batch") || strstr(argv[0],"night_control" ) ) {
    run = RUN_CONTROL;
    prefix = "image";
    object = "";
    shutter = " -s on";
  }
  if( strstr(argv[0],"night_flats") ) {
    run = RUN_FLATS;
    prefix = "f";
    object = " -name flat";
    shutter = " -s on";
  }
  if( strstr(argv[0],"night_darks") ) {
    run = RUN_DARKS;
    prefix = "d";
    object = " -name dark";
    shutter = " -s off";
  }

  cout << COOLVER << endl;

  char *h = getenv(NIGHTVIEW_HOST);
  if( h )
    host = " -host "+string(h);

  if( argc <= 1 ) {
    show_help(prefix,object,run,point_rad,point_tol);
    return(0);
  }

  /* processing of command line parameters */
  for(int i = 1; i < argc; i++ ) {

    int j;

    if( strcmp(argv[i],"-host") == 0 && i++ <= argc ) {
      host = " -host "+string(argv[i]);
    }

    else if( (strcmp(argv[i],"-h") == 0) || (strcmp(argv[i],"--help") == 0) ) {
      show_help(prefix,object,run,point_rad,point_tol);
      return(0);
    }
    
    else if( strcmp(argv[i],"--version") == 0 ) {
      cout << COOLVER << endl;
      return(0);
    }
     
    else if( strcmp(argv[i],"-oo") == 0 && i++ <= argc ) {
      oname = argv[i];
    }
     
    else if( strcmp(argv[i],"-o") == 0 && i++ <= argc ) {
      prefix = argv[i];
    }

    else if( strcmp(argv[i],"-n") == 0 && i++ <= argc && 
	     sscanf(argv[i],"%d\n",&j) == 1  ) {
      number = j;
    }

    else if( run != RUN_DARKS && strcmp(argv[i],"-d") == 0 && i++ <= argc && 
	     sscanf(argv[i],"%d\n",&j) == 1  ) {
      dark = j;
    }

    else if( strcmp(argv[i],"-w") == 0 && i++ <= argc && 
	     sscanf(argv[i],"%d\n",&j) == 1  ) {
      wait = j;
    }

    else if( strcmp(argv[i],"-b") == 0 && i++ <= argc ) {
      bin = " -b "+string(argv[i]);
    }

    else if( strcmp(argv[i],"-c") == 0 && i++ <= argc ) {
      chip = " -c "+string(argv[i]);
    }

    else if( strcmp(argv[i],"-r") == 0 && i++ <= argc ) {
      region = " -r '"+string(argv[i])+"'";
    }

    else if( run == RUN_CONTROL && strcmp(argv[i],"-p") == 0 && i++ <= argc ) {
      point = argv[i];
    }

    else if( run == RUN_CONTROL && strcmp(argv[i],"-pr") == 0 && i++ <= argc ) {
      point_rad = argv[i];
    }

    else if( run == RUN_CONTROL && strcmp(argv[i],"-pp") == 0 && i++ <= argc ) {
      point_tol = argv[i];
    }
    
    else if( strcmp(argv[i],"-v") == 0 ) { 
      nightview = true;
    }

    else if( ( strcmp(argv[i],"-name") == 0 || strcmp(argv[i],"-object") == 0)
	     && i++ <= argc ) {
      object = " -name '"+string(argv[i])+"'";
    }

    else if((strcmp(argv[i],"-obsname")==0 || strcmp(argv[i],"-observer")== 0)
	     && i++ <= argc ) {
      observer = " -obsname '"+string(argv[i])+"'";
    }

    else {
      char *w;
      vector<string> a;
      for( w = strtok(argv[i],","); w; w = strtok(NULL,",") )
	a.push_back(string(w));
      if( run == RUN_DARKS && a.size() == 1 ) {
	Exp_exp e(prefix,shutter,a[0]);
	exp.push_back(e);
      }
      else if( (run == RUN_FLATS || run == RUN_CONTROL) ) {
	if( a.size() == 2 ) {
	  Exp_exp e(prefix,shutter,a[0],a[1]);
	  exp.push_back(e);
	}
	else if( a.size() == 1 ) {
	  Exp_exp e(prefix,shutter,a[0],"");
	  exp.push_back(e);
	}
	else {
	  clog << "An exposure or filter specification not recognized." << endl;
	  return(1);
	}

      }
      else {
	clog << "An exposure or filter specification not recognized." << endl;
	return(1);
      }
    }
  }

  if( exp.empty() ) {
    cout << "No exposue time and filter specified." << endl;
    return(1);
  }
    
  if( run == RUN_CONTROL && (object == "" || observer == "") ) {
    cout << "An object and an observer specification is required." << endl;
    return(1);
  }

  // construct the output filename format
  for(vector<Exp_exp>::iterator e = exp.begin(); e != exp.end(); ++e) {
    if( oname )
      e->set_format(oname);
    else
      e->set_format(number);
  }

  //  startup the camera
  if( !debug && popen2("night_power login"+host) == ""  ) {
    clog << "Connection to camera failed. Exiting..." << endl;
    return(1);
  }

  // startup graphical window
  FILE *xw = NULL;
  if( nightview ) {
    xw = popen("xnightview - >& /dev/null","w");
  }

  // save current telescope coordinates
  if( ! point.empty() ) {
    string a = popen2("telescope get -ra -dec"+host);
    if( ! a.empty() ) {
      bool last = false;
      for(string::iterator c = a.begin(); c != a.end(); c++)
	if( *c == '\n' ) {
          if( last ) {
	    last = false;
	    *c = ',';
	  }
	  else
	    *c = ' ';
	}
	else if( *c == ' ' )
	  a.erase(c);
      point_coo = a;
    }
  }

  for(int i = 1; i <= number; i++ ) {
    
    for(vector<Exp_exp>::iterator e = exp.begin(); e != exp.end();++e) {

      string name = e->name(i);

      e->print_status(name);

      string dname;
      if( (run != RUN_DARKS) && dark > 0 && (i % dark == 0) )
	dname = make_dark(e->exp_time(),bin+region+chip+object+observer+host);

      e->select_filter(chip+host);

      e->exposure(name,bin+region+chip+object+observer+host);

      if( dark > 0 && dname != "" ) 
	dark_subtract(dname,name);

      if( xw ) {
	fprintf(xw,"%s\n",name.c_str());
	fflush(xw);
      }
      cout << "Finished" << endl;

      if( point != "" )
	pointing(name,point,point_rad,point_tol,point_coo);

      if( wait > 0 && i < number) { 
        cout << "\rWaiting for " << wait << " second(s).\r";
	cout.flush();
        sleep(wait);
      }
    }
  }

  if( xw ) {
    cout << "Press [Enter] to end...";
    char enter[1];
    cin.getline(enter,1);
    pclose(xw);
  }

  return(0);
}


string popen2(const string& com)
{
  string o;
  int c;

  if( debug )
    cout << com << endl;

  FILE *f = popen(com.c_str(),"r");
  if( f ) {
    while( (c = fgetc(f)) != EOF )
	o += char(c);
    pclose(f);
  }
  return(o);
}    

void system2(const string& com)
{
  if( debug )
    cout << com << endl;
  system(com.c_str());
}

unsigned defn0(const string& c)
{
  if( debug )
    cout << c << endl;

  unsigned n;
  stringstream oss;
  oss << popen2(c);
  oss >> n;
  return(n);
}


string make_dark(string exp, string opt)
{
  stringstream oss;
  oss << "tmpdark" << exp << ".fits";
  string dname = oss.str();
  
  cout << "DarkExposure = " << exp << " sec\n" << endl;
  system2("night_exposure -p -t "+exp+" -s off -o "+dname+" -object dark"+opt);

  return(dname);
}

void dark_subtract(string dname, string name)
{
  string temp = "dark_" + name;

  system2("night_dark "+temp+" "+name+" "+dname);
  system2("chmod -w "+name);
}

void pointing(string name, string point, string point_rad, string point_tol,
	      string point_coo)
{
  stringstream oss;
  string radif, decdif;
  double dx,dy,dr;
  
  cout << "\rPointing ...\r";
  oss << popen2("night_pointer "+point+" "+name+" "+point_rad);
  oss >> radif >> decdif;
  if( sscanf(radif.c_str(),"%lf",&dx) == 1 && 
      sscanf(decdif.c_str(),"%lf",&dy) == 1 &&
      sscanf(point_tol.c_str(),"%lf",&dr) == 1 && 
      fabs(dx) > 1e-13 && fabs(dy) > 1e-13 &&
      sqrt(dx*dx + dy*dy) > dr/3600.0 ) {

    // slew to desires point
    system2("telescope -q -vra 0.05 -vdec 0.05 set +ra "+radif+" +dec "+decdif);
    // synchronise coordinates
    if( ! point_coo.empty() )
      system2("telescope set -cal "+point_coo);
  }
  
}  

void show_help(const string& prefix, const string& object, int run, 
	       const string& point_rad, const string& point_tol)
{
  string obj;

  if( object != "" ) 
    obj = ", default: "+object;

  if( run == RUN_DARKS ) {
    cout << "night_darks - non-iteractive dark snapshoting with CCD camera" << endl;
    cout << "Usage: night_darks [options] exptime1 [exptime2] ..." << endl;
  }
  else if( run == RUN_FLATS ) {
    cout << "night_flats - non-iteractive flat snapshoting with CCD camera" << endl;
  cout << "Usage: night_flats [options] exptime1,filter1 [exptime2,filter2] ..." << endl;
  }
  else {
    cout << "night_batch - non-interactive snapshoting with CCD camera" << endl;
  cout << "Usage: night_control [options] exptime1,filter1 [exptime2,filter2] ..." << endl;
  }
  cout << "\t-oo output filename mask, default: ''" << endl;
  cout << "\t-o output prefix name, default: '"+prefix+"'" << endl;
  cout << "\t-n number of exposures, default=0" << endl;
  cout << "\t-obsname,-observer observer's name" << endl;
  cout << "\t-name,-object object's name" << obj << endl;
  if( run != RUN_DARKS )
    cout << "\t-d auto-dark substract, dark interval, default: 0 no darks" << endl;
  cout << "\t-w wait for specified num. of seconds, integer, default: 1"<<endl;
  cout << "\t-b binning, default=3" << endl;
  cout << "\t-r select region by format: (x1,y1,x2,y2), default: full area" << endl;
  cout << "\t-c select chip, I - imaging (default), T - tracking" << endl;
  cout << "\t-v view exposed image with xnightview" << endl;
  if( run == RUN_CONTROL ) {
    cout << "\t-p <x,y> point mode" << endl;
    cout << "\t-pr radius to locate pointer star on image in pixels (default: "+point_rad+")" << endl;
    cout << "\t-pp toleration for pointer mode in arc seconds (default: "+point_tol+")" << endl;
  }
  cout << "\t-host address:port internet address server, default: local connect" << endl;
  cout << "\t-h print help" << endl;
  cout << endl;
  cout << "Setting of environment variable NIGHTVIEW_HOST is equivalent to -host option." << endl;

  if( run == RUN_CONTROL || run == RUN_FLATS ) {
    string filters = popen2("night_filter -list 2>/dev/null");
    cout << "List of possible filters: " << filters << endl; 
  }
}
