
/*

  night exposure

     start exposure, download and save an image.

  $Id: night_exposure.c,v 1.40 2008-02-23 23:00:24 hroch Exp $

*/

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
#include <signal.h>
#include <sys/wait.h>
#include "nightview.h"
#include "commands.h"
#include "ccdtypes.h"
#include "ccd.h"
#include "ccdcommon.h"
#include "fitsadd.h"

#define OUTNAME "nightview.fits"

#define BITPIX 16

static CCD *ccd;

void show_help();
void interrupt();


int main(int argc, char *argv[])
{
  int i,ret, dt;
  char *output = OUTNAME;
  char *tempname;
  char *objectname = NULL;
  char *obsname = NULL;
  char *host = NIGHT_LOCALHOST;
  int mode = -1;
  float t, exptime, p;
  double minexp, maxexp;
  int shutter,binning, chip,x1,y1,x2,y2,prs,prsn,l=0,l0=0;

  /* set up interrupt  signal handler */
  signal(SIGINT,interrupt);
  signal(SIGTERM,interrupt);

  host = getenv(NIGHTVIEW_HOST);
  if( host == NULL )
    host = NIGHT_LOCALHOST;

  if( argc == 1 ) {
    show_help();
    exit(0);
  }

  /* default values */
  exptime = -1;
  shutter = 1;
  x1 = -1;
  binning = 3;
  chip = 0;
  prs = 0;
  prsn = 0;

  /* processing of command line parameters */
  for( i = 1; i < argc; i++ ) {

    /* help */
    if( (strcmp(argv[i],"-h") == 0) || (strcmp(argv[i],"--help") == 0) ) {
      show_help();
      return(0);
    }

    /* version */
    if( (strcmp(argv[i],"--version") == 0) ) {
      printf("%s\n",COOLVER);
      return(0);
    }

    /* exsposure time */
    if( (strcmp(argv[i],"-t") == 0 && i++ < argc ) ) {
      ret = sscanf(argv[i],"%f",&exptime) == 1;
    }

    /* shutter */
    if( (strcmp(argv[i],"-s") == 0 && i++ <= argc ) ) {
      if( strcmp(argv[i],"on") == 0 )
	shutter = 1;
      else if( strcmp(argv[i],"off") == 0 )
	shutter = 2;
      else
	shutter = 0;
    }

    /* binning */
    if( (strcmp(argv[i],"-b") == 0 && i++ <= argc ) ) {
      if( strcmp(argv[i],"1") == 0 )
	binning = 1;
      else if( strcmp(argv[i],"2") == 0 )
	binning = 2;
      else if( strcmp(argv[i],"3") == 0 )
	binning = 3;
      else	
	binning = 3;
    }

    /* progress indicator */
    if( (strcmp(argv[i],"-p") == 0 ) ) {
      prs = 1;
    }

    if( (strcmp(argv[i],"-pn") == 0 ) ) {
      prsn = 1;
    }

    /* readout mode */
    if( (strcmp(argv[i],"-m") == 0 && i++ <= argc ) ) {
      if( sscanf(argv[i],"%d",&mode) != 1)
	mode = -1;
    }

    /* select chip */
    if( (strcmp(argv[i],"-c") == 0 && i++ <= argc ) ) {
      if( strcmp(argv[i],"I") == 0 )
	chip = 0;
      else if( strcmp(argv[i],"T") == 0 )
	chip = 1;
      else
	chip = 1;
    }

    /* region */
    if( (strcmp(argv[i],"-r") == 0 && i++ <= argc ) )
      if( sscanf(argv[i],"%*c%d%*c%d%*c%d%*c%d%*c",&x1,&y1,&x2,&y2) != 4) {
	fprintf(stderr,"Unrecognized format of region\n");
	fprintf(stderr,"The '(1,2,100,300)' means (x1,y1) = (1,2) and (x2,y2) = (100,300)\n");
	return(1);
      }

    /* output filename */
    if( (strcmp(argv[i],"-o") == 0 && i++ <= argc ) )
      output = strdup(argv[i]);

    /* object name */
    if( (strcmp(argv[i],"-name") == 0 && i++ <= argc ) )
    	objectname = argv[i];
    if( (strcmp(argv[i],"-object") == 0 && i++ <= argc ) )
    	objectname = argv[i];

    /* observer's name */
    if( (strcmp(argv[i],"-obsname") == 0 && i++ <= argc ) )
    	obsname = argv[i];
    if( (strcmp(argv[i],"-observer") == 0 && i++ <= argc ) )
    	obsname = argv[i];

    /* set server address */
    if( strcmp(argv[i],"-host") == 0 && i++ <= argc ) {
      host = argv[i];
    }

  }

  if( exptime < 0 ) {
    fprintf(stderr,"Exposute time not specified. Abort.\n");
    return(1);
  }

  if( (ccd = ccd_init(host)) == NULL) {
    fprintf(stderr,"Camera init at %s failed.\n",host);
    return(1);
  }

  if( ! ccd_connected(ccd) ) {
    fprintf(stderr,"Camera at %s not connected.\n",host);
    ccd_free(ccd);
    return(1);
  }

  minexp = ccd_get_minexp(ccd,chip);
  maxexp = ccd_get_maxexp(ccd,chip);
  if( exptime < minexp || exptime > maxexp ){
    fprintf(stderr,"Exposue time out of range %lf ... %lf.\n",minexp,maxexp);
    ccd_free(ccd);
    return(1);
  }

  /* start exposure */
  if( exp_start(ccd,exptime,shutter,chip) != 1 ) {
    fprintf(stderr,"Start exposure error.\n");
    ccd_free(ccd);
    return(1);
  }

  /* in meantime, setup parametres */
  if( obsname )
    ccd_def_observer(ccd,obsname);
  else
    ccd_def_observer(ccd,"");
  if( objectname )
    ccd_def_object(ccd,objectname);
  else
    ccd_def_object(ccd,"");

  /* wait for end of exposure */
  if( exptime < 10.0 )
    dt = 1;
  else if( exptime >= 10.0 )
    dt = 2;
  else
    dt = 1;

  if( prs ) {
    l0 = printf("Exposing ");
    l = printf("0/%.1f",exptime);
  }
  for( t = 0.0; t < exptime+2.0*dt && exp_stat(ccd,chip) == RUN; t += dt ) {
    if( prs ) {
      for(i=0; i< l;i++)
	printf("\b");
      l = printf("%d/%.1f",(int)(t+0.5),exptime);
      fflush(stdout);
    }
    if( prsn ) {
      fprintf(stdout,"Exposing %d/%.1f\n",(int)(t+0.5),exptime);
      fflush(stdout);
    }
    sleep(dt);
  }
  if( prs ) {
    for( i = 0; i < l+l0; i++ )
      printf("\b");
  }

  /* be sure that exposure is over */
  if( ! exp_stop(ccd,chip) ) {
    fprintf(stderr,"Stop exposure failure.\n");
    ccd_free(ccd);
    return(1);
  }    

  if( x1 < 0 ) {
    if( mode < 0 )
      mode = binning - 1;
    x1 = 1;
    y1 = 1;
    x2 = ccd_get_readout_width(ccd,chip,mode);
    y2 = ccd_get_readout_height(ccd,chip,mode); 
  }

  /* start readout */
  if( read_start(ccd,BITPIX,mode,x1,y1,x2,y2,chip) != 1 ) {
    fprintf(stderr,"Start readout failure.\n");
    ccd_free(ccd);
    return(1);
  }

  /* wait for end of readout, but not more than 90 sec */
  if( prs ) {
    l0 = printf("Downloading ");
    l = printf("0%%");
  }
  for( t = 0; t < 90 && (p = read_stat(ccd)) < 100.0; t = t + 1 ) {
    if( prs ) {
      for(i=0; i < l;i++)
	printf("\b");
      l = printf("%.1f%%",p);
      fflush(stdout);
    }
    if( prsn ) {
      fprintf(stdout,"Downloading %.1f%%\n",p);
      fflush(stdout);
    }
    usleep(100000);
  }
  if( prs ) {
    for( i = 0; i < l+l0; i++ )
      printf("\b");
    for( i = 0; i < l+l0; i++ )
      printf(" ");
    for( i = 0; i < l+l0; i++ )
      printf("\b");
  }

  if( read_stat(ccd) < 100.0 || ! (tempname = read_download(ccd,output)) ) {
    fprintf(stderr,"Readout exposure error\n");
    ccd_free(ccd);
    return(1);
  }

  /* remove temp file on server */
  if( ! read_remove(ccd) )
    fprintf(stderr,"Remove of temporary file %s failed.\n",tempname);

  ccd_free(ccd);

  return(0);
}

/*---------------------------------------------------------------------*/

void show_help(void)
{
  printf("%s\n",COOLVER);
  printf("night_exposure - make a single exposure on a camera\n");
  printf("Usage: night_exposure [-t time] [-s shutter on|off] [-b 1|2|3] [-o output]\n");
  printf("options:\n");
  printf("\t -t    exposure time in seconds\n");
  printf("\t -s    shutter state, on - open, off - close for dark frame\n");
  printf("\t -r    select region by format: (x1,y1,x2,y2), default: full area\n");
  printf("\t -b    binning, 1 - 1x1, 2 - 2x2, 3 - 3x3, default: 3 \n");
							 
  printf("\t -c    select chip, I - imaging (default), T - tracking\n");
  printf("\t -m    readout mode (see output of 'night_power info')\n");
  printf("\t -o    name of an output file, default to %s\n",OUTNAME);
  printf("\t -name object's name\n");
  printf("\t -obsname observer's name\n");
  printf("\t -p    show progress indicator\n");
  printf("\t -pn   show progress indicator (on new line)\n");
  printf("\t -host address:port internet address server, default: local connect\n");
  printf("Use Ctrl-C to interrupt the exposure or download in progress.\n");
  printf("Setting of environment variable %s is equivalent to -host option.\n",
	 NIGHTVIEW_HOST);
  printf("\n");
}

void interrupt(int signal)
{
  fprintf(stderr,"Interrupt signal received. Terminating exposure in progress.\n");
  exp_stop(ccd,0);
  read_stop(ccd);
  ccd_free(ccd);
  exit(1);
}
