
/*

   Simple FITS image analyzer, print out difference between
   required and actual position of selected star on image in degrees.
  
   Part of nightview package.

   $Id: night_pointer.cpp,v 1.4 2008-10-29 22:04:07 hroch Exp $

*/

#include <cmath>
#include <string>
#include <cstring>
#include <iostream>
#include <stdexcept>
#include <fitsio.h>


using namespace std;

const int THRESHOLD = 25;


void show_help();

int qmed(int n, int *a, int k);


int main(int argc, char *argv[])
{
  double x0,y0,focus,psize;

  /* process command line parameters */
  if( argc < 3 ) {
    show_help();
    return(0);
  }

  if( sscanf(argv[1],"%lf,%lf",&x0,&y0) != 2 ) {
    show_help();
    return(0);
  }

  //cout << x0 << " " << y0 << argv[2] << endl;

  try {

    /* open image */
    fitsfile *f;
    int status = 0;
    long naxes[2];
    char comment[80];
    fits_open_file(&f, argv[2], READONLY, &status);

    fits_read_key(f,TDOUBLE,(char *)"FOCUS",&focus,comment,&status);
    fits_read_key(f,TDOUBLE,(char *)"XPIXSIZE",&psize,comment,&status);
    if( status != 0 ) throw invalid_argument("File or KEY not found");

    fits_get_img_size(f,2,naxes,&status);
    int dim = naxes[0]*naxes[1];
    
    int* d = new int[dim];
    int fpixel = 1, nullval = 0, ii;
  
    fits_read_img(f, TINT, fpixel, dim, &nullval, d, &ii, &status);
    fits_close_file(f, &status);
    if( status != 0 ) throw invalid_argument("Data input error");

    /* compute scale, [pix/deg] */
    double c = (psize*1e-6)/focus*57.29577951;

    /* locate maximum */
    int radius = 30;
    if( argc > 3 ) {
      if( sscanf(argv[3],"%d",&radius) != 1 )
	radius = 30;
    }

    // locate maximum
    int xmax = 0, ymax = 0, mmax = 0;
    int i0 = int(x0+0.5);
    int j0 = int(y0+0.5);
    for(int i = i0 - radius; i < i0 + radius;  i++)
      for(int j = j0 - radius; j < j0 + radius;  j++) {
	int n = j*naxes[0] + i;
	if( 0 <= n && n < dim && d[n] > mmax ) {
	  mmax = d[n];
	  xmax = i;
	  ymax = j;
	}
      }

    // compute background noise
    int ndim = (2*radius+1)*(2*radius+1);
    int *data = new int[ndim];
    int nd = 0;

    for(int i = i0 - radius; i < i0 + radius;  i++)
      for(int j = j0 - radius; j < j0 + radius;  j++) {
	int n = j*naxes[0] + i;
	if( 0 <= n && n < dim && nd < ndim ) {
	  data[nd] = d[n];
	  nd = nd + 1;
	}
      }
    int mean = qmed(nd,data,nd/2);

    nd = 0;
    for(int i = i0 - radius; i < i0 + radius;  i++)
      for(int j = j0 - radius; j < j0 + radius;  j++) {
	int n = j*naxes[0] + i;
	if( 0 <= n && n < dim && nd < ndim ) {
	  data[nd] = abs(d[n] - mean);
	  nd = nd + 1;
	}
      }
    int mad = qmed(nd,data,nd/2);
    delete[] data;

    
    // compute center
    /*
    double xc=0,yc=0;
    double w = 0;
    for(int i = xmax - 5; i < xmax + 5; i++)
      for(int j = ymax - 5; j < ymax + 5; j++) {
	double f = d[j*naxes[0] + i];
	xc = xc + f*i;
	yc = yc + f*j;
	w = w + f;
      }
    if( w > 1e-10 ) {
      xc = xc/w;
      yc = yc/w;
    }
    else {
      xc = xmax;
      yc = ymax;
    }
    */

    // compute sharpness
    
    // compute round

    //    free(d);
    delete[] d;

    double dx = - (xmax - x0)*c;
    double dy = (ymax - y0)*c;

    //cout << mad << " " << mean << " " << mmax << endl;
    if( mmax > THRESHOLD*mad + mean ) {
      
      if( fabs(dx) > 0.5*c )
	cout << dx;
      else
	cout << "0";

      cout << " ";

      if( fabs(dy) > 0.5*c )
	cout << dy;
      else
	cout << "0";

    }
    else
      cout << "0 0";

    cout << endl;
    return(0);
  }

  catch (invalid_argument a) {
    
    cout << "0 0" << endl;
    return(1);
  }

}

void show_help()
{
  cout << "Usage: night_pointer x0,y0 image [search radius]" << endl;
}

int qmed(int n, int *a, int k)
{
  int w,x;
  int l,r,i,j;

  x = 0;

  l = 0;
  r = n - 1;

  while( l < r ) {
    x = a[k];
    i = l;
    j = r;

    do {

      while( a[i] < x )
        i++;
      while( x < a[j] )
        j--;

      if( i <= j ) {
        w = a[i];
        a[i] = a[j];
        a[j] = w;
        i++;
        j--;
      }
      
    } while ( i <= j );

    if( j < k ) l = i;
    if( k < i ) r = j;
  }

  return(x);

}

