
/* Command line frontend to ccd client library

   night_temperature
   
       controls cooling of CCD camera
       set/get temperature and cooling fan state

  $Id: night_temperature.c,v 1.15 2008-02-23 15:39:18 hroch Exp $

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "nightview.h"
#include "ccdtypes.h"
#include "ccd.h"
#include "ccdcommon.h"

int main(int argc, char *argv[])
{
  CCD *ccd;
  int i, sw, get = 0, set = 0;
  float temp;
  char *host = NIGHT_LOCALHOST;
  void show_help(void);

  host = getenv(NIGHTVIEW_HOST);
  if( host == NULL )
    host = NIGHT_LOCALHOST;

  if( argc == 1 ) {
    show_help();
    return(0);
  }
  
  /* processing of command line parameters */
  for( i = 1; i < argc; i++ ) {

    if( (strcmp(argv[i],"-h") == 0) || (strcmp(argv[i],"--help") == 0) ) {
      show_help();
      return(0);
    }
    else if( (strcmp(argv[i],"--version") == 0) ) {
      printf("%s\n",COOLVER);
      return(0);
    }
    /* get info */
    else if( strcmp(argv[i],"get") == 0 ) {
      get = i + 1;
    }
    /* set parameters */
    else if( strcmp(argv[i],"set") == 0 ) {
      set = i + 1;
    }
    /* set server address */
    else if( strcmp(argv[i],"-host") == 0 && i++ <= argc ) {
      host = argv[i];
    }
    /*
    else
      fprintf(stderr,"Unrecognized option: %s\n",argv[i]);
    */

  }

  if( (set && get) || (! set && !get ) ) {
    fprintf(stderr,"Please, specify only 'get' or 'set', not both or nothing.\n");
    return(1);
  }

  if( (ccd = ccd_init(host)) == NULL ) {
    fprintf(stderr,"Camera init at %s failed.\n",host);
    return(1);
  }

  if( ! ccd_connected(ccd) ) {
    fprintf(stderr,"Camera at %s not connected.\n",host);
    ccd_free(ccd);
    return(1);
  }
    

  /* print informations */
  if( get ) {

    sw = 0;
    for( i = 1; i < argc; i++ ) {

      if( strcmp(argv[i],"-t") == 0) {
	sw = 1;
	printf("%5.1f\n",temp_ccd(ccd)); 
      }

      else if( strcmp(argv[i],"-ta") == 0) {
	printf("%5.1f\n",temp_air(ccd)); 
	sw = 1;
      }

      else if( strcmp(argv[i],"-ts") == 0) {
	printf("%5.1f\n",temp_setpoint(ccd)); 
	sw = 1;
      }

      else if( strcmp(argv[i],"-r") == 0) {
	printf("%5.1f\n",temp_regul(ccd)); 
	sw = 1;
      }

      else if( strcmp(argv[i],"-f") == 0) {
	printf("%3s\n","on");
	sw = 1;
      }
      
    }

    if( sw == 0 ) {
      printf("ccd temperature = %5.1f\n",temp_ccd(ccd));
      printf("air temperature = %5.1f\n",temp_air(ccd));
      printf("temperature setpoint = %5.1f\n",temp_setpoint(ccd));
      printf("cooling power   = %5.1f\n",temp_regul(ccd));
      printf("fan status      = %3s\n","on");
    }

  } /* get */

  /* set cooling */
  if( set ) {

    sw = 0;
    for( i = 1; i < argc; i++ ) {

      temp = -999.9;
      if( strcmp(argv[i],"-t") == 0 && i++ <= argc ) {
	sw = 1;
	if( sscanf(argv[i],"%f",&temp) == 1 && temp > -999.0 ) {
	  if( temp_set(ccd,temp) != 1) {
	    fprintf(stderr,"Temperature not set\n");
	    ccd_free(ccd);
	    return(1);
	  }
	}
      }
      else if( strcmp(argv[i],"-off") == 0 ) {
	temp_off(ccd);
	sw = 1;
      }
    }

    if( sw == 0 ) {
      fprintf(stderr,"Nothing to set.\n");
      ccd_free(ccd);
      return(1);
    }

  }

  ccd_free(ccd);
  return(0);
}

/*---------------------------------------------------------------------*/

void show_help(void)
{
  printf("%s\n",COOLVER);
  printf("night_temperature - a camera temperature \n");
  printf("Usage: night_temperature [get|set] [-t temperature] [-off] [-host address]\n");
  printf("options\n");
  printf("\t get:\n");
  printf("\t\t -t   print ccd current temperature in degrees of Celsius\n");
  printf("\t\t -ta  print air current temperature in degrees of Celsius\n");
  printf("\t\t -ts  print temperature setpoint in degrees of Celsius\n");
  printf("\t\t -r   print current cooling power (relatively, in percents)\n");
  printf("\t\t -f   print current status of fan, on or off\n");
  printf("\t\t  without switches print all available cooling information\n");
  printf("\t set:\n");
  printf("\t\t -t temperature   set temperature of a cooler\n");
  printf("\t\t -off             set regulation of temperature to off\n\n");
  printf("\t -host address:port internet address server, local connect is default\n");
  printf("Setting of environment variable %s is equivalent to -host option.\n"
	 ,NIGHTVIEW_HOST);
  printf("\n");
  /*
  printf("Note, it is advised to turn off regulation, but leave the fan on,\n");
  printf("for a minute or so prior to shutting down the camera.\n");
  */
}
