
/*

  Xmove

  $Id: xmove.cpp,v 1.26 2008-02-17 16:18:29 hroch Exp $

*/


// For compilers that support precompilation, includes "wx/wx.h".
#include "wx/wxprec.h"

#ifdef __BORLANDC__
    #pragma hdrstop
#endif

// the application icon (under Windows and OS/2 it is in resources and even
// though we could still include the XPM here it would be unused)
#if !defined(__WXMSW__) && !defined(__WXPM__)
    #include "xmove-icon.xpm"
#endif

// for all others, include the necessary headers (this file is usually all you
// need because it includes almost all "standard" wxWidgets headers)
#ifndef WX_PRECOMP
    #include "wx/wx.h"
#endif
#include <wx/tokenzr.h>
#include <wx/wfstream.h>
#include <wx/txtstrm.h>
#include <wx/tglbtn.h>
#include <wx/stdpaths.h>
#include <wx/gbsizer.h>
#include <wx/propdlg.h> 
#include <wx/generic/propdlg.h>
#include <wx/tooltip.h>

#include "xmove.h"
#include "nightview.h"
#include "yes.xpm"
#include "no.xpm"

IMPLEMENT_APP(XMove)

BEGIN_EVENT_TABLE(MoveFrame, wxFrame)
     EVT_CLOSE(MoveFrame::OnClose)
     EVT_SIZE(MoveFrame::OnSize)
     EVT_TIMER(ID20, MoveFrame::OnUpdate)
     EVT_TIMER(ID18, MoveFrame::OnSlew)
     EVT_END_PROCESS(ID16, MoveFrame::EndCooUpdate)
     EVT_END_PROCESS(ID17, MoveFrame::OnEndSlew)
     EVT_MENU(ID41, MoveFrame::Open)
     EVT_MENU(ID42, MoveFrame::Exit)
     EVT_MENU(ID43, MoveFrame::Calibrate)
     EVT_BUTTON(ID43, MoveFrame::Calibrate)
     EVT_MENU(ID44, MoveFrame::Parking)
     EVT_MENU(ID45, MoveFrame::About)
     EVT_MENU(ID46, MoveFrame::Preferences)
     EVT_MENU(ID47, MoveFrame::SlewOn)
     EVT_MENU(ID48, MoveFrame::ClockOnOff)
     EVT_MENU(ID71, MoveFrame::Up)
     EVT_MENU(ID72, MoveFrame::Down)
     EVT_MENU(ID73, MoveFrame::Right)
     EVT_MENU(ID74, MoveFrame::Left)
     EVT_RADIOBOX(ID28, MoveFrame::Clock)
     EVT_RADIOBOX(ID29, MoveFrame::Step)
     EVT_COMBOBOX(ID21, MoveFrame::ComboObject)
     EVT_BUTTON(ID22, MoveFrame::GetObject)
     EVT_TEXT_ENTER(ID21, MoveFrame::GetObject)
     EVT_TOGGLEBUTTON(ID25, MoveFrame::Slew)
     EVT_BUTTON(ID31, MoveFrame::Up)
     EVT_BUTTON(ID32, MoveFrame::Down)
     EVT_BUTTON(ID33, MoveFrame::Right)
     EVT_BUTTON(ID34, MoveFrame::Left)
     EVT_COMMAND_SCROLL(ID26, MoveFrame::Velra)
     EVT_COMMAND_SCROLL(ID27, MoveFrame::Veldec)
     EVT_END_PROCESS(ID81, MoveFrame::OnEndCalibrate)
     EVT_END_PROCESS(ID82, MoveFrame::OnEndUp)
     EVT_END_PROCESS(ID83, MoveFrame::OnEndDown)
     EVT_END_PROCESS(ID84, MoveFrame::OnEndLeft)
     EVT_END_PROCESS(ID85, MoveFrame::OnEndRight)
     EVT_END_PROCESS(ID86, MoveFrame::OnEndClockSwitch)
END_EVENT_TABLE()

bool XMove::OnInit()
{
  for(int i = 0; i < argc; i++) {
    wxString c(argv[i]);

    if( c.Cmp(_("-h")) == 0 || c.Cmp(_("--help")) == 0 ) {
      wxString cp(COPYLEFT,wxConvUTF8);
      wxString ver(NIGHTVERSION,wxConvUTF8);
      wxPrintf(_("XMove %s, %s\nNightView interactive control center for telescope.\n"),
	       ver.c_str(),cp.c_str());
      return(0);
    }
  }

  MoveFrame *mframe = new MoveFrame(_("xmove"));
  mframe->Show(true);

  for(int i = 0; i < argc; i++) {
    wxString c(argv[i]);
    
    if( c.Cmp(_("-c")) == 0 && i++ < argc )
      mframe->OpenCatalogue(argv[i]);
  }
  
  return true;
}

void MoveFrame::OnClose(wxCloseEvent& event)
{
  Destroy();
}

void MoveFrame::OnSize(wxSizeEvent& event)
{
  event.Skip();

  if( GetStatusBar()->GetFieldsCount() > 1 ) {
    wxRect rect;
    GetStatusBar()->GetFieldRect(1, rect);
    progress->SetSize(rect);
  }
}

MoveFrame::MoveFrame(wxString title):
  wxFrame(NULL, wxID_ANY, title, wxDefaultPosition, wxDefaultSize,
  	  wxDEFAULT_FRAME_STYLE), UpdateTimer(this,ID20), SlewTimer(this,ID18)
{
  SetIcon(wxICON(xmove_icon));

  conf = new MoveConfig(_("xmoverc"));


  // init preferences file
  //  InitPrefer(sp.GetUserConfigDir()+_("/.xmoverc"));

  // init object file
  wxStandardPaths sp;
  wxArrayString objs;
  catalogue = new Catalogue();
  catalogue->Load(sp.GetUserConfigDir()+_("/.xmove.stars"),true);
  catalogue->Objects(objs);

  dstep = 1.0/60.0;
  InitMaxVel(maxvra,maxdec);
  vra = maxvra;
  vdec = maxdec;
  clockup = InitClock();

  // move
  wxMenu *menuFile = new wxMenu;
  menuFile->Append(ID41, _T("&Open\tAlt-O"),_T("Open a star catalogue"));
  menuFile->Append(ID46, _T("&Preferences..."),_T("Set up"));
  menuFile->AppendSeparator();
  menuFile->Append(ID42, _T("&Exit\tAlt-Q"),_T("Xmove exit. The telescope itself is untouched."));

  menuAct = new wxMenu;
  menuAct->Append(ID43,_T("Calibrate\tAlt-C"),_("Calibrate to"));
  menuAct->Append(ID44,_T("Park\tAlt-P"),_("Park telescope"));
  menuAct->Append(ID47,_T("Slew\tAlt-S"),_("Slew to"));
  menuAct->Append(ID48,_T("Clock On/Off\tAlt-T"),_("Hourly motion On/Off"));
  menuAct->AppendSeparator();
  menuAct->Append(ID71,_T("Step Up\tAlt-U"),_("Step up"));
  menuAct->Append(ID72,_T("Step Down\tAlt-D"),_("Step down"));
  menuAct->Append(ID73,_T("Step Right\tAlt-R"),_("Step right"));
  menuAct->Append(ID74,_T("Step Left\tAlt-L"),_("Step down"));

  wxMenu *menuHelp = new wxMenu;
  menuHelp->Append(ID45, _T("&About..."), _T("Xmove basic info"));

  wxMenuBar *menuBar = new wxMenuBar();
  menuBar->Append(menuFile, _T("&File"));
  menuBar->Append(menuAct, _T("&Action"));
  menuBar->Append(menuHelp, _T("&Help"));

  SetMenuBar(menuBar);

  wxBoxSizer *topsizer = new wxBoxSizer(wxVERTICAL);

  // input objectname
  wxBoxSizer *osizer = new wxBoxSizer(wxHORIZONTAL);
  wxStaticBoxSizer *os = new wxStaticBoxSizer(wxHORIZONTAL,this,_(" Object "));
  cb = new wxComboBox(this,ID21,_(""),wxDefaultPosition,wxDefaultSize,objs,
		      wxCB_DROPDOWN|wxCB_SORT,wxDefaultValidator,_("combo"));
  os->Add(cb,wxSizerFlags(3).Expand().Border(wxALL, 2));

  // get coordinates button
  gobj = new wxButton(this,ID22,_T(" Get coo. "));
  os->Add(gobj);
  osizer->Add(os,wxSizerFlags(1).Expand().Border(wxALL, 2));
  topsizer->Add(osizer,wxSizerFlags().Expand());

  // setup coordinates  
  wxBoxSizer *ssizer = new wxBoxSizer(wxHORIZONTAL);
  wxStaticBoxSizer *ss = new wxStaticBoxSizer(wxHORIZONTAL,this,_(" Set coordinates "));
  calibr = new wxButton(this,ID43,_(" Calibrate "));
  ss->Add(calibr,wxSizerFlags().Expand().Border(wxALL, 2));
  wxStaticText *txt_ra = new wxStaticText(this,wxID_ANY,_(" R.A.: "));
  ss->Add(txt_ra,wxSizerFlags(1).Expand().Border(wxALL, 2).Align(wxALIGN_CENTER_VERTICAL|wxALIGN_CENTER_HORIZONTAL));
  alfa = new wxTextCtrl(this,ID23,_(" 0:0:0 "));
  ss->Add(alfa,wxSizerFlags(2).Expand().Border(wxALL, 2));
  wxStaticText *txt_dec = new wxStaticText(this,wxID_ANY,_(" Decl.: "));
  ss->Add(txt_dec,wxSizerFlags(1).Expand().Border(wxALL, 2).Align(wxALIGN_CENTER_VERTICAL));
  delta = new wxTextCtrl(this,ID24,_(" 0:0:0 "));
  ss->Add(delta,wxSizerFlags(2).Expand().Border(wxALL, 2));
  slewto = new wxToggleButton(this,ID25,_(" Slew to "));
  ss->Add(slewto,wxSizerFlags().Expand().Border(wxALL, 2));
  ssizer->Add(ss,wxSizerFlags(1).Expand().Border(wxALL, 2));
  topsizer->Add(ssizer,wxSizerFlags().Expand());

  // current coordinates
  wxStaticBoxSizer *ct = new wxStaticBoxSizer(wxHORIZONTAL,this,_(" Telescope coordinates "));
  wxGridBagSizer *t = new wxGridBagSizer(1,10);
  t->SetFlexibleDirection(wxVERTICAL);
  wxSizerFlags tflags = wxSizerFlags().Expand().Align(wxALIGN_CENTER);
  wxStaticText *tt_ra = new wxStaticText(this,wxID_ANY,_(" Right Ascension:"));
  wxStaticText *tt_dec = new wxStaticText(this,wxID_ANY,_(" Declination:"));
  wxStaticText *tt_ara = new wxStaticText(this,wxID_ANY,_(" Apparent R. A.:"));
  wxStaticText *tt_adec = new wxStaticText(this,wxID_ANY,_(" Apparent Decl.:"));
  wxStaticText *tt_az = new wxStaticText(this,wxID_ANY,_(" Azimuth:"));
  wxStaticText *tt_al = new wxStaticText(this,wxID_ANY,_(" Altitude:"));
  wxStaticText *tt_mhc = new wxStaticText(this,wxID_ANY,_(" Local sider. time:"));
  wxStaticText *tt_ha = new wxStaticText(this,wxID_ANY,_(" Hour Angle: "));
  wxStaticText *tt_jd = new wxStaticText(this,wxID_ANY,_(" Julian Date (UT):"));
  wxStaticText *tt_syn = new wxStaticText(this,wxID_ANY,_(" Calibrated? "));
  ra = new wxTextCtrl(this,ID50,_(""),wxDefaultPosition,wxDefaultSize,wxTE_READONLY|wxTE_RIGHT);
  dec = new wxTextCtrl(this,ID51,_(""),wxDefaultPosition,wxDefaultSize,wxTE_READONLY|wxTE_RIGHT);
  ara = new wxTextCtrl(this,ID52,_(""),wxDefaultPosition,wxDefaultSize,wxTE_READONLY|wxTE_RIGHT);
  adec = new wxTextCtrl(this,ID53,_(""),wxDefaultPosition,wxDefaultSize,wxTE_READONLY|wxTE_RIGHT);
  az = new wxTextCtrl(this,ID54,_(""),wxDefaultPosition,wxDefaultSize,wxTE_READONLY|wxTE_RIGHT);
  al = new wxTextCtrl(this,ID55,_(""),wxDefaultPosition,wxDefaultSize,wxTE_READONLY|wxTE_RIGHT);
  ha = new wxTextCtrl(this,ID56,_(""),wxDefaultPosition,wxDefaultSize,wxTE_READONLY|wxTE_RIGHT);
  mhc = new wxTextCtrl(this,ID57,_(""),wxDefaultPosition,wxDefaultSize,wxTE_READONLY|wxTE_RIGHT);
  jd = new wxTextCtrl(this,ID58,_(""),wxDefaultPosition,wxDefaultSize,wxTE_READONLY|wxTE_RIGHT);
  synchro = new wxStaticBitmap(this,wxID_ANY,wxBitmap(no_xpm));

  wxBoxSizer *ssyn = new wxBoxSizer(wxHORIZONTAL);
  ssyn->Add(tt_syn);
  ssyn->Add(synchro);

  t->Add(tt_ra,wxGBPosition(0,0));
  t->Add(ra,wxGBPosition(0,1),wxDefaultSpan,wxEXPAND);
  t->Add(tt_dec,wxGBPosition(0,2));
  t->Add(dec,wxGBPosition(0,3),wxDefaultSpan,wxEXPAND);

  t->Add(tt_ara,wxGBPosition(1,0));
  t->Add(ara,wxGBPosition(1,1),wxDefaultSpan,wxEXPAND);
  t->Add(tt_adec,wxGBPosition(1,2));
  t->Add(adec,wxGBPosition(1,3),wxDefaultSpan,wxEXPAND);

  t->Add(tt_az,wxGBPosition(2,0));
  t->Add(az,wxGBPosition(2,1),wxDefaultSpan,wxEXPAND);
  t->Add(tt_al,wxGBPosition(2,2));
  t->Add(al,wxGBPosition(2,3),wxDefaultSpan,wxEXPAND);

  t->Add(tt_ha,wxGBPosition(3,0));
  t->Add(ha,wxGBPosition(3,1),wxDefaultSpan,wxEXPAND);
  t->Add(tt_mhc,wxGBPosition(3,2));
  t->Add(mhc,wxGBPosition(3,3),wxDefaultSpan,wxEXPAND);

  t->Add(ssyn,wxGBPosition(4,0));
  t->Add(tt_jd,wxGBPosition(4,1));
  t->Add(jd,wxGBPosition(4,2),wxGBSpan(1,2),wxEXPAND);

  ct->Add(t,wxSizerFlags().Expand());
  topsizer->Add(ct,wxSizerFlags().Expand());

  // velocity setup
  wxSizerFlags vflags = wxSizerFlags(1).Expand();
  wxBoxSizer *vsizer = new wxBoxSizer(wxHORIZONTAL);
  wxStaticBoxSizer *vs = new wxStaticBoxSizer(wxHORIZONTAL,this,_(" Velocity "));
  wxStaticText *txt_vra = new wxStaticText(this,wxID_ANY,_(" R.A.(%): "));
  vs->Add(txt_vra,wxSizerFlags(1).Expand());
  velra = new wxSlider(this,ID26,100,1,100,wxDefaultPosition,wxDefaultSize,wxSL_HORIZONTAL|wxSL_LABELS|wxSL_AUTOTICKS|wxSL_BOTTOM);
  vs->Add(velra,wxSizerFlags(2).Expand());
  wxStaticText *txt_vdec = new wxStaticText(this,wxID_ANY,_(" Decl.(%): "));
  vs->Add(txt_vdec,wxSizerFlags(1).Expand());
  veldec = new wxSlider(this,ID27,100,1,100,wxDefaultPosition,wxDefaultSize,wxSL_HORIZONTAL|wxSL_LABELS|wxSL_AUTOTICKS|wxSL_BOTTOM);
  vs->Add(veldec,wxSizerFlags(2).Expand());
  vsizer->Add(vs,wxSizerFlags(1).Expand());
  topsizer->Add(vsizer,wxSizerFlags().Expand());

  // step
  wxBoxSizer *bsizer = new wxBoxSizer(wxHORIZONTAL);
  wxArrayString st;
  st.Add(_("1/3'"));
  st.Add(_("1'"));
  st.Add(_("5'"));
  step = new wxRadioBox(this,ID29,_(" Step "),wxDefaultPosition, 
			wxDefaultSize,st,1,wxRA_SPECIFY_COLS);
  step->SetSelection(1);
  bsizer->Add(step,wxSizerFlags(1).Expand());

  // direction
  wxStaticBoxSizer *ds = new wxStaticBoxSizer(wxHORIZONTAL,this,_(" Direction "));  
  wxGridSizer *d = new wxGridSizer(3);
  wxSizerFlags dflags = wxSizerFlags(1).Expand().Align(wxALIGN_CENTER);
  up = new wxButton(this,ID31,_T(" +Dec "));
  down = new wxButton(this,ID32,_T(" -Dec "));
  right = new wxButton(this,ID33,_T(" -RA "));
  left = new wxButton(this,ID34,_T(" +RA "));
  d->AddStretchSpacer();
  d->Add(up);
  d->AddStretchSpacer();
  d->Add(left);
  d->AddStretchSpacer();
  d->Add(right);
  d->AddStretchSpacer();
  d->Add(down);
  d->AddStretchSpacer();
  ds->Add(d,wxSizerFlags().Expand());

  bsizer->Add(ds,wxSizerFlags().Expand());

  // clock
  wxArrayString cl;
  cl.Add(_(" On "));
  cl.Add(_(" Off "));
  clock = new wxRadioBox(this,ID28,_(" Clock "),wxDefaultPosition, 
			wxDefaultSize,cl,1,wxRA_SPECIFY_COLS);
  clock->SetSelection(!clockup);
  bsizer->Add(clock,wxSizerFlags(1).Expand());


  topsizer->Add(bsizer,wxSizerFlags().Expand());

  CreateStatusBar(1);
  SetStatusText(_T("XMove"));
  progress = 0;

  SetSizerAndFit(topsizer);

  // tooltips
  wxToolTip::Enable(true);
  wxToolTip::SetDelay(500);
  synchro->SetToolTip(_("fsfs"));
  jd->SetToolTip(_("fsfs"));

  // startup coordinates update
  UpdateProc = 0;
  UpdateTimer.Start(conf->Udelay()*1000);
  CooUpdate();
}

void MoveFrame::OpenCatalogue(const wxString& file)
{
  Catalogue *c = new Catalogue();
  if( c->Load(file) ) {
    delete catalogue;
    catalogue = c;
    wxArrayString objs;
    catalogue->Objects(objs);
    cb->Clear();
    for(size_t i = 0; i < objs.GetCount(); i++ )
      cb->Append(objs[i]);
    cb->SetValue(_(""));
    SetStatusText(_("Catalogue ")+file+_(" loaded."));
  }
}

void MoveFrame::Open(wxCommandEvent& WXUNUSED(event))
{
  SetStatusText(_T("Open catalogue."));
  wxFileDialog select(this,_T("Choose a star catalogue file"),_(""),_(""),
		      _T("All files (*.*)|*"),wxFILE_MUST_EXIST|wxCHANGE_DIR);
  if (select.ShowModal() == wxID_OK )
    OpenCatalogue(select.GetPath());
}

void MoveFrame::Exit(wxCommandEvent& WXUNUSED(event))
{
  wxStandardPaths sp;

  // save object file
  if( catalogue->IsWritable() ) {
    wxString home = sp.GetUserConfigDir();
    SetStatusText(_("Saving catalogue to ")+home+_("/.xmove.stars"));
    catalogue->Save(home+_("/.xmove.stars"));
  }

  // save preferences
  //SavePrefer(sp.GetUserConfigDir()+_("/.xmoverc"));

  SetStatusText(_("Terminating childrens ..."));
  /*
  if( UpdateProc ) {
    UpdateProc->Detach();
  if( SlewProc )
    SlewProc->Detach();
  */

  Destroy();
}

void MoveFrame::About(wxCommandEvent& WXUNUSED(event))
{
  wxString cp(COPYLEFT,wxConvUTF8);
  wxString ver(NIGHTVERSION,wxConvUTF8);
  wxString info;

  info.Printf(_("XMove %s\nNightView interactive control center for telescope\n%s"),
	      ver.c_str(),cp.c_str());

  wxMessageBox(info,_T("About XMove"));
}

void MoveFrame::Calibrate(wxCommandEvent& WXUNUSED(event))
{

  double a = adeg(alfa->GetValue());
  double d = ddeg(delta->GetValue());

  if( a < -900.0 || d < -900.0 ) {
    SetStatusText(_("Improper input coordinates (6:04:12 or 6:04.2 or 91.05)."));
    return;
  }

  wxString t;
  t.Printf(_("Calibrating to R.A.:%.3f°, Decl.:%.3f° ..."),a,d);
  SetStatusText(t);

  wxString x;
  x.Printf(_("telescope set -cal %f %f"),a,d);

  calibr->Enable(false);
  menuAct->Enable(ID43,false);
  CalProc = new wxProcess(this,ID81);
  CalProc->Redirect();

  if( wxExecute(x,wxEXEC_ASYNC,CalProc) <= 0 )
    SetStatusText(t+_(" failed"));
}

void MoveFrame::OnEndCalibrate(wxProcessEvent& event)
{
  if( event.GetExitCode() == 0 ) {

    double a = adeg(alfa->GetValue());
    double d = ddeg(delta->GetValue());

    if( a < -900.0 || d < -900.0 ) {
      SetStatusText(_("Improper input coordinates (6:04:12 or 6:04.2 or 91.05)."));
      return;
    }

    wxString t;
    t.Printf(_("Calibrating to R.A.:%.3f°, Decl.:%.3f° ... succesfull"),a,d);
    SetStatusText(t);
    CooUpdate();
  }
  else
    SetStatusText(_("Calibration of coordinates failed."));

  delete CalProc;

  calibr->Enable(true);
  menuAct->Enable(ID43,true);
}

void MoveFrame::Parking(wxCommandEvent& WXUNUSED(event))
{

  wxString t;
  t.Printf(_("Parking to H.A.:%.3f°, Decl.:%.3f° ..."),
	   conf->ParkHA(),conf->ParkDecl());
  SetStatusText(t);

  wxString slewcom;
  slewcom.Printf(_("telescope set -coa %f %f -vra %f -vdec %f -q -pn"),
		 conf->ParkHA(),conf->ParkDecl(),vra,vdec);

  SlewProc = new wxProcess(this,ID17);
  SlewProc->Redirect();
  slewpid = wxExecute(slewcom,wxEXEC_ASYNC,SlewProc);

  if( slewpid == 0 ) {
    SetStatusText(_("Telescope slew command failed"));
    slewto->SetValue(false);
  }
  else {

    GetStatusBar()->SetFieldsCount(2);
    int wh[] = { -4, -1 }; 
    GetStatusBar()->SetStatusWidths(2,wh);
    int sbt[] = { wxSB_NORMAL, wxSB_FLAT };
    GetStatusBar()->SetStatusStyles(2,sbt);
    wxRect rect;
    GetStatusBar()->GetFieldRect( 1, rect );

    progress = new wxGauge(GetStatusBar(),wxID_ANY,1,rect.GetPosition(),rect.GetSize());

    SetStatusText(_("Starting telescope parking ..."));
    slewto->Enable(false);
    clock->Enable(false);
    up->Enable(false);
    down->Enable(false);
    right->Enable(false);
    left->Enable(false);
    SlewTimer.Start(conf->Sdelay()*1000);
  }
}

void MoveFrame::ClockOnOff(wxCommandEvent& WXUNUSED(event))
{
  int c = clock->GetSelection();
  int cc;
  if( c == 0 )
    cc = 1;
  else
    cc = 0;

  ClockSwitch(cc == 0);
  clock->SetSelection(cc);  
}

void MoveFrame::Clock(wxCommandEvent& WXUNUSED(event))
{ 
  ClockSwitch(clock->GetSelection() == 0);
}

void MoveFrame::ClockSwitch(bool enable)
{
  clock->Enable(false);
  if( enable ) {
    SetStatusText(_("Setting clock on..."));

    ClockProc = new wxProcess(this,ID86);
    ClockProc->Redirect();    

    if( wxExecute(_T("telescope set -c on"),wxEXEC_ASYNC,ClockProc) <= 0 )
      SetStatusText(_("Setting clock on... failed"));
  }
  else {
    SetStatusText(_("Setting clock off..."));

    ClockProc = new wxProcess(this,ID86);
    ClockProc->Redirect();

    if( wxExecute(_T("telescope set -c off"),wxEXEC_ASYNC,ClockProc) <= 0 )
      SetStatusText(_("Setting clock on... failed"));
  }
}

void MoveFrame::OnEndClockSwitch(wxProcessEvent& event)
{
  if( event.GetExitCode() == 0 ) {
    SetStatusText(_("Setting clock ... done"));
  }
  else {
    SetStatusText(_("Setting clock ... failed"));
    clock->SetSelection(1);
  }

  delete ClockProc;
  clock->Enable(true);
}


void MoveFrame::Step(wxCommandEvent& WXUNUSED(event))
{
  int s = step->GetSelection();

  if( s == 0 ) {
    dstep = 0.33333333/60.0;
    SetStatusText(_("Step changed to 1/3'."));
  }
  else if( s == 1 ) {
    dstep = 1.0/60.0;
    SetStatusText(_("Step changed to 1'."));
  }
  else if( s == 2 ) {
    dstep = 5.0/60.0;
    SetStatusText(_("Step changed to 5'."));
  }
  else {
    dstep = 10.0/60.0;
    SetStatusText(_("Step changed to 10'."));
  }
}

void MoveFrame::Up(wxCommandEvent& WXUNUSED(event))
{
  wxString t(_("Increasing Declination ..."));
  SetStatusText(t);

  up->Enable(false);
  down->Enable(false);
  slewto->Enable(false);
  menuAct->Enable(ID71,false);

  wxString x;
  x.Printf(_("telescope set +dec %f -vdec %f"),dstep,vdec);

  UpProc = new wxProcess(this,ID82);
  UpProc->Redirect();

  if( wxExecute(x,wxEXEC_ASYNC,UpProc) <= 0 )
    SetStatusText(t+_(" failed"));
}

void MoveFrame::OnEndUp(wxProcessEvent& event)
{
  if( event.GetExitCode() == 0 ) {
    SetStatusText(_("Increasing Declination ... done"));
    CooUpdate();
  }
  else
    SetStatusText(_("Increasing Declination ... failed"));

  delete UpProc;
  up->Enable(true);
  down->Enable(true);
  slewto->Enable(true);
  menuAct->Enable(ID71,false);
}


void MoveFrame::Down(wxCommandEvent& WXUNUSED(event))
{
  wxString t(_("Decreasing Declination ..."));
  SetStatusText(t);

  up->Enable(false);
  down->Enable(false);
  slewto->Enable(false);
  menuAct->Enable(ID72,false);
  
  wxString x;
  x.Printf(_("telescope set +dec %f -vdec %f"),-dstep,vdec);

  DownProc = new wxProcess(this,ID83);
  DownProc->Redirect();

  if( wxExecute(x,wxEXEC_ASYNC,DownProc) <= 0 )
    SetStatusText(t+_(" failed"));
}

void MoveFrame::OnEndDown(wxProcessEvent& event)
{
  if( event.GetExitCode() == 0 ) {
    SetStatusText(_("Decreasing Declination ... done"));
    CooUpdate();
  }
  else
    SetStatusText(_("Decreasing Declination ... failed"));

  delete DownProc;
  up->Enable(true);
  down->Enable(true);
  slewto->Enable(true);
  menuAct->Enable(ID72,true);
}

void MoveFrame::Right(wxCommandEvent& WXUNUSED(event))
{
  wxString t(_("Decreasing Right Ascension ..."));
  SetStatusText(t);

  right->Enable(false);
  left->Enable(false);
  slewto->Enable(false);
  menuAct->Enable(ID73,false);
  
  wxString x;
  x.Printf(_("telescope set +ra %f -vra %f"),-dstep,vra);

  RightProc = new wxProcess(this,ID85);
  RightProc->Redirect();

  if( wxExecute(x,wxEXEC_ASYNC,RightProc) <= 0 )
    SetStatusText(t+_(" failed"));
}

void MoveFrame::OnEndRight(wxProcessEvent& event)
{
  if( event.GetExitCode() == 0 ) {
    SetStatusText(_("Decreasing Right Ascension ... done"));
    CooUpdate();
  }
  else
    SetStatusText(_("Decreasing Right Ascension ... failed"));

  delete RightProc;
  right->Enable(true);
  left->Enable(true);
  slewto->Enable(true);
  menuAct->Enable(ID73,true);
}


void MoveFrame::Left(wxCommandEvent& WXUNUSED(event))
{
  wxString t(_("Increasing Right Ascension ..."));
  SetStatusText(t);

  right->Enable(false);
  left->Enable(false);
  slewto->Enable(false);
  menuAct->Enable(ID74,false);
  
  wxString x;
  x.Printf(_("telescope set +ra %f -vra %f"),dstep,vra);

  LeftProc = new wxProcess(this,ID84);
  LeftProc->Redirect();

  if( wxExecute(x,wxEXEC_ASYNC,LeftProc) <= 0 )
    SetStatusText(t+_(" failed"));

}

void MoveFrame::OnEndLeft(wxProcessEvent& event)
{
  if( event.GetExitCode() == 0 ) {
    SetStatusText(_("Increasing Right Ascension ... done"));
    CooUpdate();
  }
  else
    SetStatusText(_("Increasing Right Ascension ... failed"));

  delete LeftProc;
  right->Enable(true);
  left->Enable(true);
  slewto->Enable(true);
  menuAct->Enable(ID74,true);
}

void MoveFrame::Velra(wxScrollEvent& WXUNUSED(event))
{
  double p = double(velra->GetValue())/100.0;
  vra = p*maxvra;
  
  wxString t;
  t.Printf(_("Right Ascension velocity changed to %.2f°/s."),vra);
  SetStatusText(t);
}

void MoveFrame::Veldec(wxScrollEvent& WXUNUSED(event))
{
  double p = double(veldec->GetValue())/100.0;
  vdec = p*maxdec;
  
  wxString t;
  t.Printf(_("Declination velocity changed to %.2f°/s."),vdec);
  SetStatusText(t);
}



//-------------------------------------------------------------------

// sixdeg and adeg,ddeg converts a string to real number by this way:
//
// if (there is more than one number, first is integer, separated by space,
//     :,\t\n\r ):
//    the value is in sexadecimal format: (first+second/60+third/3600),
//    the third may be ommited, second may be real
// if ( there is only one number readable as real):
//    the value is coordinate in degrees (not multiplied by 15 for RA)
// else
//    the value is not readable as a coordinate
//
// optional minus sign may be presented anywhere before number, that is
//    value -0:30 leads to -0.5

double MoveFrame::sixdeg(wxString a)
{
  bool minus = a.Find('-') > -1;
  double x = 0.0;
  double six = 1.0;
  int nitems = 0;

  wxStringTokenizer ta(a,_(" :\t\r\n"),wxTOKEN_STRTOK);

  // hours/degs
  if( ta.HasMoreTokens() ) {
    wxString t = ta.GetNextToken();
    long h;
    if( t.ToLong(&h) )
      x = labs(h);
    else
      return -999.0;
    nitems++;
  }
  
  // minutes, seconds
  six = 60.0;
  while( ta.HasMoreTokens() ) {
    wxString t = ta.GetNextToken();
    double f;
    if( t.ToDouble(&f) )
      x = x + fabs(f)/six;
    else
      return -999.0;
    nitems++;
    six = six*60.0;
 }

  if( minus )
    x = -x;

  if( nitems >= 2  )
    return x;
  else
    return -999.0;
}

double MoveFrame::adeg(wxString a)
{
  double f = sixdeg(a);
  double epsilon = 1e-10;
  if( 0.0-epsilon <= f && f <= 24.0+epsilon )
    return 15.0*f;

  if( a.ToDouble(&f) )
    return f;

  return -999.0;
}

double MoveFrame::ddeg(wxString a)
{
  double f = sixdeg(a);
  double epsilon = 1e-10;
  if( -90.0-epsilon <= f && f <= 90.0+epsilon )
    return f;

  if( a.ToDouble(&f) )
    return f;

  return -999.0;
}

wxString MoveFrame::asix(double x)
{

  x = x + 0.6/3600.0; // prevent to s == 60

  if( x < 0.0 )
    x = x + 360.0;

  if( x >= 360.0 )
    x = x - 360.0;

  x = x / 15.0;

  int d = int(x);
  int m = int(60.0*(x - d));
  int s = int(3600.0*(x - (d + m/60.0)));

  wxString t;
  t.Printf(_("%d:%02d:%02d"),d,m,s);
  return t;
}

wxString MoveFrame::dsix(double x)
{
  wxString sgn;

  // prevent to s == 60
  if( x >= 0.0 )
    x = x + 0.001/3600.0;
  else
    x = x - 0.001/3600.0;

  if( x < 0.0 ) {
    sgn = _("-");
    x = -x;
  }

  int d = int(x);
  int m = int(60.0*(x - d));
  int s = int(3600.0*(x - (d + m/60.0)));

  wxString t;
  t.Printf(_("%s%d:%02d:%02d"),sgn.c_str(),d,m,s);
  return t;
}


void MoveFrame::CooUpdate(void)
{
  if( ! UpdateProc ) {

    UpdateProc = new wxProcess(this,ID16);
    UpdateProc->Redirect();
    
    wxExecute(_("telescope get"),wxEXEC_ASYNC,UpdateProc);
  }
}

void MoveFrame::EndCooUpdate(wxProcessEvent& event)
{
  //wxPrintf(_("EndCooUpdate %d %d\n"),event.GetPid(),event.GetExitCode());
  if( UpdateProc && UpdateProc->IsInputAvailable() ) {
    wxInputStream *is = UpdateProc->GetInputStream();
    wxTextInputStream text(*is);
    wxArrayString lines;
    while( ! is->Eof() )
      lines.Add(text.ReadLine());
    ShowCoordinates(lines);
    delete UpdateProc;
    UpdateProc = 0;
  }
}

void MoveFrame::ComboObject(wxCommandEvent& WXUNUSED(event))
{
  inlist = false;
  wxString o = cb->GetValue();
  int i = catalogue->FindName(o);
  if( i > -1 ) {

    wxString name = catalogue->Name(i);
    wxString at = catalogue->Alpha(i);
    wxString dt = catalogue->Delta(i);
    double a,d;
    at.ToDouble(&a);
    dt.ToDouble(&d);
    alfa->SetValue(asix(a));
    delta->SetValue(dsix(d));
    SetStatusText(_("Coordinates has been set for ")+name+_("."));
    inlist = true;
    return;
  }
}

void MoveFrame::GetObject(wxCommandEvent& event)
{
  ComboObject(event);
  if( inlist ) 
    return;

  wxString name = cb->GetValue();

  SetStatusText(_("Connecting Simbad..."));
  Simbad simbad(name);
  if( simbad.Connect() ) {

    wxString object;
    if( simbad.GetCount() == 1 )
      object = simbad.Item(0);
    else {
      wxArrayString choices;
      for(size_t i = 0; i < simbad.GetCount(); i++) {
	choices.Add(simbad.Item(i));
	//wxPrintf(_("%s\n"),simbad.Item(i).c_str());
      }

      wxSingleChoiceDialog dialog(this,_T("Multiple objects has been found"),
				  _T("Select an appropriate item."),choices);
      if (dialog.ShowModal() == wxID_OK) {
	object = dialog.GetStringSelection();
	SetStatusText(object+_("has been selected"));
      }
      else
	SetStatusText(_("No object selected."));
    }

    if( ! object.empty() ) {

      //wxPrintf(_("%s\n"),object.c_str());
      wxStringTokenizer ta(object,_("\t"));
      wxString objname = ta.GetNextToken();
      double a = adeg(ta.GetNextToken());
      double d = ddeg(ta.GetNextToken());
      alfa->SetValue(asix(a));
      delta->SetValue(dsix(d));
      wxString t;
      t.Printf(_(" %.3f°, %.3f°"),a,d);
      SetStatusText(_("Coordinates for ")+name+_(":")+t+_("."));

      // save object to list
      wxString l;
      l.Printf(_("%s\t%f\t%f"),name.c_str(),a,d);
      catalogue->Add(l);
      cb->Append(name);
      cb->SetStringSelection(name);
    }
  }
  else {
    SetStatusText(_("Connection to Simbad failed or object not found."));
  }
}

void MoveFrame::OnUpdate(wxTimerEvent& event)
{
  CooUpdate();
}

void MoveFrame::SlewOn(wxCommandEvent& event)
{
  if( ! slewto->GetValue() ) {
    slewto->SetValue(true);
    Slew(event);
  }
  else {
    slewto->SetValue(false);
    Slew(event);
  }
}

void MoveFrame::Slew(wxCommandEvent& WXUNUSED(event))
{
  //wxPrintf(_("Slew %d\n"),slewto->GetValue());
  if( slewto->GetValue() ) {

    double a = adeg(alfa->GetValue());
    double d = ddeg(delta->GetValue());

    if( a < -900.0 || d < -900.0 ) {
      SetStatusText(_("Improper input coordinates (6:04:12 or 6:04.2 or 91.05)."));
      return;
    }

    wxString x;
    x.Printf(_("telescope set -coo %f %f -vra %f -vdec %f -q -pn"),a,d,vra,vdec);
    slewcom = x;


    /*
    wxGauge g(this,wxID_ANY,100,wxPoint(50,50),wxSize(200,20));
    g.SetValue(50);
    g.Show();
    */
    //progress->SetValue(50);


    SlewProc = new wxProcess(this,ID17);
    SlewProc->Redirect();
    slewpid = wxExecute(slewcom,wxEXEC_ASYNC,SlewProc);
    if( slewpid == 0 ) {
      SetStatusText(_("Telescope slew command failed"));
      slewto->SetValue(false);
    }
    else {

      GetStatusBar()->SetFieldsCount(2);
      int wh[] = { -4, -1 }; 
      GetStatusBar()->SetStatusWidths(2,wh);
      int sbt[] = { wxSB_NORMAL, wxSB_FLAT };
      GetStatusBar()->SetStatusStyles(2,sbt);
      wxRect rect;
      GetStatusBar()->GetFieldRect( 1, rect );

      progress = new wxGauge(GetStatusBar(),wxID_ANY,1,rect.GetPosition(),rect.GetSize());

      SetStatusText(_("Starting telescope slew ..."));
      clock->Enable(false);
      up->Enable(false);
      down->Enable(false);
      right->Enable(false);
      left->Enable(false);
      slewto->SetLabel(_("Stop slew"));
      SlewTimer.Start(conf->Sdelay()*1000);
    }
  }
  else{
    if( SlewProc ) {
      SlewProc->Kill(slewpid,wxSIGTERM);
    }
  }
  
}

void MoveFrame::OnSlew(wxTimerEvent& event)
{
  //wxPrintf(_("OnSlew %s %d\n"),slewcom.c_str(),SlewTimer.IsRunning());
  if( SlewProc && SlewProc->IsInputAvailable() ) {
    wxInputStream *is = SlewProc->GetInputStream();
    wxTextInputStream text(*is);
    if( ! is->Eof() ) {
      wxString line = text.ReadLine();
      SetStatusText(line);
      line.Replace(_("/"),_(" "));
      line.Replace(_("["),_(""));
      line.Replace(_("]"),_(""));
      line.Replace(_("Right Ascension"),_(""));
      line.Replace(_("Declination"),_(""));
      wxStringTokenizer ta(line,_(" "));
      int i = 0;
      double tcur = 0,tha = 0,tdec = 0;
      while( ta.HasMoreTokens() ) {
	wxString t = ta.GetNextToken();
	if( ! t.empty() ) {
	  double s;
	  t.ToDouble(&s);

	  if( i == 0 )
	    tcur = s;
	  else if( i == 1 )
	    tha = s;
	  else if( i == 3 )
	    tdec = s;
	  i++;
	}
      }
      if( progress ) {
	if( tdec > tha )
	  progress->SetRange(int(tdec));
	else
	  progress->SetRange(int(tha));
	progress->SetValue(int(tcur));
      }
    }
  }
}

void MoveFrame::OnEndSlew(wxProcessEvent& event)
{
  //wxPrintf(_("OnEndSlew %d %d\n"),event.GetPid(),SlewTimer.IsRunning());
  SlewTimer.Stop();
  if( event.GetExitCode() == 0 )
    SetStatusText(_("Telescope slew done."));
  else
    SetStatusText(_("Telescope slew failed."));

  GetStatusBar()->SetFieldsCount(1);
  int wh[] = { -1 }; 
  GetStatusBar()->SetStatusWidths(1,wh);
  int sbt[] = { wxSB_NORMAL };
  GetStatusBar()->SetStatusStyles(1,sbt);
  delete progress;
  progress = 0;
  delete SlewProc;
  SlewProc = 0;
  slewpid = 0;

  slewto->SetLabel(_(" Slew to "));
  slewto->SetValue(false);
  clock->Enable(true);
  up->Enable(true);
  down->Enable(true);
  right->Enable(true);
  left->Enable(true);

  // end of parking
  if( ! slewto->IsEnabled() ) {
    slewto->Enable(true);
    ClockSwitch(false);
    clock->SetSelection(1);
  }
    
  //wxPrintf(_("OnEndSlew ... end\n"));

}

void MoveFrame::InitMaxVel(double& vra, double& vdec)
{
  vra = 1.0;
  vdec = 1.0;

  wxArrayString x;
  if( wxExecute(_T("telescope get -velstat"),x) == 0 && x.GetCount() > 0 ) {
    wxString line = x.Item(0);
    wxStringTokenizer ta(line,_(" /"));
    int n = 0;
    while( ta.HasMoreTokens() ) {
      wxString t = ta.GetNextToken();
      if( n == 1 ) {
	t.ToDouble(&vra);
      }
      else if( n == 3 ) {
	t.ToDouble(&vdec);
      }
      n++;
    }
  }
}

bool MoveFrame::InitClock()
{
  wxArrayString x;
  if( wxExecute(_("telescope get -status"),x) == 0 && x.GetCount() > 0 ) {
    wxString line = x.Item(0);
    wxStringTokenizer ta(line,_(" "));
    if( ta.HasMoreTokens() ) {
      wxString t = ta.GetNextToken();
      return t.StartsWith(_("1"));
    }
  }
  return false;
}


void MoveFrame::ShowCoordinates(const wxArrayString& text)
{
  bool synchr = false;

  for(size_t i = 0; i < text.GetCount(); i++ ) {

    wxStringTokenizer ta(text.Item(i),_("="));
    wxString key,value;
    if( ta.HasMoreTokens() )
      key = ta.GetNextToken();
    if( ta.HasMoreTokens() )  
      value = ta.GetNextToken();
    double v;
    value.ToDouble(&v);
    long k;
    value.ToLong(&k);

    if( key.StartsWith(_("Right Ascension")) )
      ra->SetValue(asix(v));
    
    else if( key.StartsWith(_("Declination")) )
      dec->SetValue(dsix(v));

    else if( key.StartsWith(_("Azimuth")) )
      az->SetValue(value+_("°"));

    else if( key.StartsWith(_("Altitude")) )
      al->SetValue(value+_("°"));
      
    else if( key.StartsWith(_("Apparent R.A.")) )
      ara->SetValue(value+_("°"));

    else if( key.StartsWith(_("Apparent Dec")) )
      adec->SetValue(value+_("°"));

    else if( key.StartsWith(_("Hour angle")) )
      ha->SetValue(value+_("°"));

    else if( key.StartsWith(_("Siderical time")) )
      mhc->SetValue(value+_(" h"));

    else if( key.StartsWith(_("Julian date")) )
      jd->SetValue(value+_(" d"));

    else if( key.StartsWith(_("Synchronized")) )
      synchr = k == 1;
  }

  if( synchr )
    synchro->SetBitmap(wxBitmap(yes_xpm));
  else
    synchro->SetBitmap(wxBitmap(no_xpm));
}

void MoveFrame::Preferences(wxCommandEvent& WXUNUSED(event))
{
  PrefFrame *pref = new PrefFrame(this->GetParent(),_(" Xmove: Preferences "),
				  wxDEFAULT_DIALOG_STYLE|wxRESIZE_BORDER);

  pref->set_udelay(conf->Udelay());
  pref->set_sdelay(conf->Sdelay());
  pref->set_local(conf->Localhost());
  pref->set_host(conf->Remotehost());
  pref->set_auth(conf->AuthKey());
  pref->set_port(conf->Port());
  pref->set_ha(conf->ParkHA());
  pref->set_decl(conf->ParkDecl());

  if( pref->ShowModal() == wxID_OK ) {
    
    conf->Set_Udelay(pref->get_udelay());
    conf->Set_Sdelay(pref->get_sdelay());
    conf->Set_Localhost(pref->get_local());
    conf->Set_Remotehost(pref->get_host());
    conf->Set_AuthKey(pref->get_auth());
    conf->Set_Port(pref->get_port());
    conf->Set_ParkHA(pref->get_ha());
    conf->Set_ParkDecl(pref->get_decl());

    UpdateTimer.Stop();
    UpdateTimer.Start(conf->Udelay()*1000);

    connection_setup(conf->Localhost(),conf->Remotehost(),
		     conf->Port(),conf->AuthKey());
    /*
    wxPrintf(_("%d %d\n"),udelay,sdelay);
    wxPrintf(_("%s %d %s %s\n"),remotehost.c_str(),
	     port,user.c_str(),pass.c_str());
    wxPrintf(_("%f %f\n"),park_ha,park_decl);
    */
  }
  pref->Destroy();
}

#define UDELAY     _("Self update delay")
#define SDELAY     _("Slew update delay")
#define LOCALHOST  _("Local_host")
#define REMOTEHOST _("Remote_host")
#define AUTHKEY    _("AuthKey")
#define PORT       _("Port")
#define PARKHA     _("Parking hour angle")
#define PARKDECL   _("Parking declination")


//void MoveFrame::InitPrefer(const wxString& rcfile)
MoveConfig::MoveConfig(wxString name): wxConfig(name)
{
  Read(UDELAY,&udelay,1);
  Read(SDELAY,&sdelay,1);
  Read(LOCALHOST,&localhost,true);
  Read(REMOTEHOST,&remotehost,_(""));
  Read(AUTHKEY,&authkey,_(""));
  Read(PORT,&port,7666l);
  Read(PARKHA,&park_ha,0.0);
  Read(PARKDECL,&park_decl,0.0);

  /*
  localhost = true;
  udelay = 1;
  sdelay = 1;
  remotehost = _("");
  user = _("");
  port = 7666;
  pass = _("");
  park_ha = 0;
  park_decl = 0;

  if( ! wxFile::Exists(rcfile.c_str()) ) 
    return;
  
  wxFileInputStream finput(rcfile);
  if( ! finput.IsOk() ) {
    return;
  }
  wxTextInputStream par(finput);
  while( ! finput.Eof() ) {
    wxStringTokenizer ta(par.ReadLine(),_("="));
    wxString key,value,val;
    if( ta.HasMoreTokens() )
      key = ta.GetNextToken();
    if( ta.HasMoreTokens() )  
      val = ta.GetNextToken();

    if( val.Find(_("'")) >= 0 ) {
      wxString v;      
      wxStringTokenizer tb(val,_("'"));
      for(int i = 0; i < 2 && tb.HasMoreTokens(); i++)
	v = tb.GetNextToken();
      if( v.Trim().IsEmpty() )
	value = _("");
      else
	value = v;
    }
    else
      value = val;

    double v;
    value.ToDouble(&v);

    if( key.StartsWith(_("Self update delay")) ) {
      long l;
      value.ToLong(&l);
      udelay = l;
    }

    if( key.StartsWith(_("Slew update delay")) ) {
      long l;
      value.ToLong(&l);
      sdelay = l;
    }

    if( key.StartsWith(_("Local host")) ) {
      localhost = value.Find(_("true")) >= 0;
    }

    if( key.StartsWith(_("Remote host")) )
      remotehost = value;

    if( key.StartsWith(_("User")) )
      user = value;

    if( key.StartsWith(_("Password")) )
      pass = value;

    if( key.StartsWith(_("Port")) ) {
      long l;
      value.ToLong(&l);
      port = l;
    }

    if( key.StartsWith(_("Parking hour angle")) ) {
      value.ToDouble(&park_ha);
    }

    if( key.StartsWith(_("Parking declination")) ) {
      value.ToDouble(&park_decl);
    }
  }
  connection_setup(localhost,remotehost,port,user,pass);
  */
}

//void MoveFrame::SavePrefer(const wxString& rcfile)
MoveConfig::~MoveConfig()
{
  SetUmask(0077);
  Write(UDELAY,udelay);
  Write(SDELAY,sdelay);
  Write(LOCALHOST,localhost);
  Write(REMOTEHOST,remotehost);
  Write(AUTHKEY,authkey);
  Write(PORT,port);
  Write(PARKHA,park_ha);
  Write(PARKDECL,park_decl);
  

  /*
  wxFileOutputStream fout(rcfile);
  if( ! fout.IsOk() )
    return;
  wxTextOutputStream par(fout);

  wxString ver(NIGHTVERSION,wxConvUTF8);

  par << _("# Xmove preferences (ver. ") << ver << _(") ") << endl;
  par << _("Self update delay") << _(" = ") << udelay << endl;
  par << _("Slew update delay") << _(" = ") << sdelay << endl;
  if( localhost )
    par << _("Local host = true") << endl;
  else
    par << _("Local host = false") << endl;
  par << _("Remote host") << _(" = '") << remotehost << _("'") << endl;
  par << _("User") << _(" = '") << user << _("'") << endl;
  par << _("Password") << _(" = '") << pass << _("'") << endl;
  par << _("Port") << _(" = ") << port << endl;
  par << _("Parking hour angle") << _(" = ") << park_ha << endl;
  par << _("Parking declination") << _(" = ") << park_decl << endl;
  */
}


void MoveFrame::connection_setup(bool localhost, wxString remotehost,
				int port, wxString auth)
{
  if( localhost )
    wxUnsetEnv(_("TELESCOPE_HOST"));
  else {
    wxString tele_host;
    if( auth.IsEmpty() )
      tele_host.Printf(_("%s:%d"),remotehost.c_str(),port);
    else
      tele_host.Printf(_("%s@%s:%d"),
		       auth.c_str(),remotehost.c_str(),port);
    wxSetEnv(_("TELESCOPE_HOST"),tele_host.c_str());
  }
}

bool Catalogue::Load(const wxString& file, bool w)
{
  writable = w;

  if( ! ::wxFileExists(file) )
    return false;

  wxFileInputStream finput(file);
  if( ! finput.IsOk() ) {
    wxString l(_("Catalogue name")+file+_(" not found."));
    ::wxLogMessage(l.c_str());
    return false;
  }
  wxTextInputStream objects(finput);
  while( ! finput.Eof() ) {
    wxString l = objects.ReadLine();
    if( finput.LastRead() == 0 ) 
      break;
    if( ! l.empty() )
	Add(l);
  }
  return true;
}

bool Catalogue::Save(const wxString& file)
{
  if( ! writable )
    return true;

  wxFileOutputStream fout(file);
  if( ! fout.IsOk() ) {
    wxString l(_("Catalogue name")+file+_(" failed to open (write to)."));
    ::wxLogError(l.c_str());
    return false;
  }
  wxTextOutputStream objects(fout);

  for(size_t i = 0; i < GetCount(); i++)
    objects << Item(i) << endl;

  return true;
}

wxString Catalogue::Name(size_t n)
{
  Select(n);
  return name1;
}

wxString Catalogue::Alpha(size_t n)
{
  Select(n);
  return alpha1;
}

wxString Catalogue::Delta(size_t n)
{
  Select(n);
  return delta1;
}

void Catalogue::Select(size_t n)
{
  if( ! (0 <= n && n < GetCount()) ) {
    name1.Empty();
    alpha1.Empty();
    delta1.Empty();
  }    

  int nn = n;
  if( nn != iitem ) {
    iitem = n;
    wxString line = Item(n);
    wxStringTokenizer ta(line,_("\t"));
    if( ta.HasMoreTokens() )
      name1 = ta.GetNextToken();
    else
      name1.Empty();
    if( ta.HasMoreTokens() )
      alpha1 = ta.GetNextToken();
    else
      alpha1.Empty();
    if( ta.HasMoreTokens() )
      delta1 = ta.GetNextToken();
    else
      delta1.Empty();
  }
}

void Catalogue::Objects(wxArrayString& objs)
{
  for(size_t i = 0; i < GetCount(); i++)
    objs.Add(Name(i));
}

int Catalogue::FindName(wxString name)
{
  for(size_t i = 0; i < GetCount(); i++)
    if( name == Name(i) )
      return i;
  return -1;
}


