
/*

  Xnightview

  $Id: control.cpp,v 1.30 2008-02-23 23:06:15 hroch Exp $

*/


// For compilers that support precompilation, includes "wx/wx.h".
#include "wx/wxprec.h"

#ifdef __BORLANDC__
    #pragma hdrstop
#endif

// for all others, include the necessary headers (this file is usually all you
// need because it includes almost all "standard" wxWidgets headers)
#ifndef WX_PRECOMP
    #include "wx/wx.h"
#endif

// the application icon (under Windows and OS/2 it is in resources and even
// though we could still include the XPM here it would be unused)
#if !defined(__WXMSW__) && !defined(__WXPM__)
    #include "camera.xpm"
#endif

#include <wx/propdlg.h>
#include <wx/spinctrl.h>
#include <wx/tglbtn.h>
#include <wx/list.h>
#include <wx/filename.h>
#include <wx/event.h>
#include <wx/txtstrm.h>
#include <wx/stdpaths.h>
#include <wx/tokenzr.h>
#include <wx/txtstrm.h>
#include <wx/wfstream.h>
#include <wx/datstrm.h>
#include <wx/tooltip.h>

#include "xnightview.h"

#define BWIDTH 3

BEGIN_EVENT_TABLE(NightControl, wxFrame)
  EVT_CLOSE(NightControl::WmExit)
  EVT_MENU(File_Open,  NightControl::FileOpen)
  EVT_MENU(File_Save,  NightControl::FileSave)
  EVT_MENU(File_Exit,  NightControl::FileExit)
  EVT_MENU(Set_Preferences,  NightControl::SetPreferences)
  EVT_MENU(About_Help,  NightControl::AboutHelp)
  EVT_MENU(About_Camera,  NightControl::AboutCamera)
  EVT_MENU(About_Fits,  NightControl::AboutFits)
  EVT_MENU(About_About,  NightControl::AboutAbout)
  EVT_TOGGLEBUTTON(Power_But, NightControl::PowerButton)
  EVT_TOGGLEBUTTON(Regulate_But, NightControl::RegulateButton)
  EVT_TEXT_ENTER(Temp_spin, NightControl::TemperatureText)
  EVT_TEXT(Temp_spin, NightControl::TemperatureChange)
  EVT_TOGGLEBUTTON(Exp_But, NightControl::ExposureButton)
  EVT_RADIOBOX(Filt_But, NightControl::FilterSwitch)
  EVT_TIMER(ID11, NightControl::OnTempUpdate)
  EVT_TIMER(ID15, NightControl::OnFilterSwitch)
  EVT_TIMER(ID17, NightControl::OnExposure)
  EVT_TIMER(ID19, NightControl::OnContinue)
  EVT_END_PROCESS(ID12, NightControl::EndTempUpdate)
  EVT_END_PROCESS(ID13, NightControl::OnEndRegulate)
  EVT_END_PROCESS(ID14, NightControl::OnEndFilterSwitch)
  EVT_END_PROCESS(ID16, NightControl::OnEndExposure)
  EVT_END_PROCESS(ID18, NightControl::OnEndInfo)
END_EVENT_TABLE()

NightControl::NightControl(NightFrame *p, wxString title, NightImage *i): 
  wxFrame(NULL, wxID_ANY, title, wxDefaultPosition, wxDefaultSize,
  	  wxDEFAULT_FRAME_STYLE), ExpTimer(this,ID17), ContinueTimer(this,ID19),
  TempTimer(this,ID11), FilterTimer(this,ID15)
{
  SetIcon(wxICON(camera));

  frame = p;
  active = false;
  image = i;
  NightConfig *conf = frame->Config();

  connection_setup(conf->Localhost(),conf->Remotehost(),conf->Port(),
		   conf->AuthKey());

  setpoint = false;

  wxMenu *menuFile = new wxMenu;
  menuFile->Append(File_Open, _("&Open\tAlt-O"),_("Open FITS file"));
  menuFile->Append(File_Save, _("&Save\tAlt-S"),_("Save to FITS file"));
  menuFile->Append(Set_Preferences,_("Preferences .."),_("Show preferences window"));
  menuFile->AppendSeparator();
  menuFile->Append(File_Exit, _("&Exit\tAlt-Q"),_("Exit. The camera itself is untouched."));

  wxMenu *menuHelp = new wxMenu;
  menuHelp->Append(About_Help, _("&Help...\tAlt-H"), _("On-line help"));
  menuHelp->Append(About_Camera, _("&Camera Info...\tAlt-C"), _("Camera Info"));
  menuHelp->Append(About_Fits, _("&FITS Info...\tAlt-I"), _("FITS header"));
  menuHelp->Append(About_About, _("&About..."), _("Show about dialog"));

  wxMenuBar *menuBar = new wxMenuBar();
  menuBar->Append(menuFile, _("&File"));
  menuBar->Append(menuHelp, _("&Help"));
  SetMenuBar(menuBar);


  CreateStatusBar(2);
  int wh[] = { -2, -1 };
  GetStatusBar()->SetStatusWidths(2,wh);
  SetStatusText(_("Nightview"));

  // main layout
  controls =  new wxBoxSizer(wxVERTICAL);
  wxSizerFlags flags = wxSizerFlags().Expand();

  wxBoxSizer *tz = new wxBoxSizer(wxHORIZONTAL);

  // zoom image
  int zoom_width = conf->ZoomSize();
  int zoom_height = conf->ZoomSize();
  winzoom = new NightZoom(this,wxSize(zoom_width,zoom_height));
  tz->Add(winzoom,wxSizerFlags(1).Border(wxALL, BWIDTH));

  winprofi = new NightGraph(this,wxSize(zoom_width,zoom_height));
  tz->Add(winprofi,wxSizerFlags(1).Border(wxALL, BWIDTH));

  wxGridSizer *grid = new wxGridSizer(2, 1, 5);
  grid->Add(new wxStaticText(this, wxID_ANY, _("Value:")),
	    wxSizerFlags().Align(wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL));

  wxStaticBoxSizer *v = new wxStaticBoxSizer(wxHORIZONTAL,this);

  value = new wxStaticText(this, wxID_ANY, _("0              "));
  v->Add(value,wxSizerFlags(1).Align(wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL));

  grid->Add(v,wxSizerFlags(1).Align(wxGROW | wxALIGN_CENTER_VERTICAL));

  grid->Add(new wxStaticText(this, wxID_ANY, _("Coordinate:")),
	    wxSizerFlags().Align(wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL));

  wxStaticBoxSizer *xy = new wxStaticBoxSizer(wxHORIZONTAL,this);

  coordinate = new wxStaticText(this, wxID_ANY, _("0   0     "));
  xy->Add(coordinate,wxSizerFlags(1).Center());

  grid->Add(xy,wxSizerFlags(1).Align(wxGROW | wxALIGN_CENTER_VERTICAL));

  grid->Add(new wxStaticText(this, wxID_ANY, _("FWHM, S/N:")),
	    wxSizerFlags().Align(wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL));

  wxStaticBoxSizer *fw = new wxStaticBoxSizer(wxHORIZONTAL,this);

  fwhm = new wxStaticText(this, wxID_ANY, _("0.0    1.0"));
  fw->Add(fwhm,wxSizerFlags(1).Center());

  grid->Add(fw,wxSizerFlags(1).Align(wxGROW | wxALIGN_CENTER_VERTICAL));

  grid->Add(new wxStaticText(this, wxID_ANY, _("Background:")),
	    wxSizerFlags().Align(wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL));

  wxStaticBoxSizer *mm = new wxStaticBoxSizer(wxHORIZONTAL,this);

  mean = new wxStaticText(this, wxID_ANY, _(" 0 "));
  mm->Add(mean,wxSizerFlags(1).Center());

  grid->Add(mm,wxSizerFlags(1).Align(wxGROW | wxALIGN_CENTER_VERTICAL));

  tz->Add(grid,wxSizerFlags(2).Expand().Border(wxALL, BWIDTH));
  controls->Add(tz,flags);

  // exposure 
  wxBoxSizer *esizer = new wxBoxSizer(wxHORIZONTAL);
  wxStaticBoxSizer *es = new wxStaticBoxSizer(wxHORIZONTAL,this,_(" Exposure "));

  wxString exp;
  exp.Printf(_("%.3f"),conf->Exptime());
  etime = new wxTextCtrl(this,wxID_ANY,exp);
  etime->Enable(false);

  es->Add(etime,wxSizerFlags(1).Expand().Border(wxALL, BWIDTH));

  wxArrayString binnings;
  binnings.Add(_("1x1"));
  binnings.Add(_("2x2"));
  binnings.Add(_("3x3"));

  ebin =  new wxChoice(this,wxID_ANY,wxPoint(-1,-1),wxSize(-1,-1),binnings);
  ebin->SetSelection(conf->Binning()-1);
  ebin->Enable(false);
  es->Add(ebin,wxSizerFlags(1).Expand().Border(wxALL, BWIDTH));

  wxArrayString types;
  types.Add(_("Light"));
  types.Add(_("Dark"));
  types.Add(_("Continue"));

  etype = new wxChoice(this,wxID_ANY,wxPoint(-1,-1),wxSize(-1,-1),types);
  etype->SetSelection(conf->Exptype());
  etype->Enable(false);
  es->Add(etype,wxSizerFlags(1).Expand().Border(wxALL, BWIDTH));

  start = new wxToggleButton(this,Exp_But,_(" Start "));
  start->Enable(false);
  es->Add(start,wxSizerFlags(1).Expand().Border(wxALL, BWIDTH));
  esizer->Add(es,wxSizerFlags(1).Expand().Border(wxALL, BWIDTH));
  controls->Add(esizer,flags);

  // identification
  wxBoxSizer *isizer = new wxBoxSizer(wxHORIZONTAL);

  wxStaticBoxSizer *sobj = new wxStaticBoxSizer(wxHORIZONTAL,this,_(" Object "));

  eobj = new wxTextCtrl(this,Exp_Obj,_(""));
  sobj->Add(eobj,wxSizerFlags(1).Expand().Border(wxALL, BWIDTH));

  isizer->Add(sobj,wxSizerFlags(1).Expand().Border(wxALL, BWIDTH));

  wxStaticBoxSizer *sobs = new wxStaticBoxSizer(wxHORIZONTAL,this,_(" Observer "));
  
  eobs = new wxTextCtrl(this,Exp_Obs,conf->Observer());
  sobs->Add(eobs,wxSizerFlags(1).Expand().Border(wxALL, BWIDTH));
  isizer->Add(sobs,wxSizerFlags(1).Expand().Border(wxALL, BWIDTH));
  controls->Add(isizer,flags);

  // filters
  filters.Add(_("filter"));
  filt = new wxRadioBox(this,wxID_ANY,_(" Filter "),wxDefaultPosition, 
			wxDefaultSize,filters,0,wxRA_SPECIFY_ROWS);
  filt->Enable(false);
  controls->Add(filt,wxSizerFlags(1).Expand().Border(wxALL, BWIDTH));


  // power & temperature
  wxBoxSizer *psizer = new wxBoxSizer(wxHORIZONTAL);

  wxStaticBoxSizer *ts = new wxStaticBoxSizer(wxHORIZONTAL,this,_(" Temperature "));
  temp = new wxTextCtrl(this,Temp_spin,_("666"));
  temp->Enable(false);
  ts->Add(temp,wxSizerFlags(1).Expand().Border(wxALL, BWIDTH));
  
  regulate = new wxToggleButton(this,Regulate_But,_(" Regulate "));
  regulate->Enable(false);
  ts->Add(regulate,wxSizerFlags(1).Expand().Border(wxALL, BWIDTH));

  psizer->Add(ts,wxSizerFlags(2).Expand().Border(wxALL, BWIDTH));


  wxStaticBoxSizer *ps = new wxStaticBoxSizer(wxHORIZONTAL,this,_(" Camera "));

  power = new wxToggleButton(this,Power_But,_("  Connect  "));
  ps->Add(power,wxSizerFlags(1).Expand().Right().Border(wxALL, BWIDTH));


  psizer->Add(ps,wxSizerFlags(1).Expand().Border(wxALL, BWIDTH));
  controls->Add(psizer,flags);

  SetSizerAndFit(controls);
}

NightControl::~NightControl()
{
  // remove previously created temporary file
  if( ! name.IsEmpty() ) {
    ::wxRemoveFile(name);
  }  
}

void NightControl::FileOpen(wxCommandEvent& WXUNUSED(event))
{

  wxFileDialog select(this,_("Choose a FITS file"),dir,file,
		      _("FITS files (*.fits)|*.fits|All files (*.*)|*"),
		      wxFILE_MUST_EXIST|wxCHANGE_DIR);
  if (select.ShowModal() == wxID_OK && frame->LoadImage(select.GetPath()) ) {

    SetStatusText(select.GetPath());
  }
  else
    SetStatusText(_("Open a FITS image failed."));

}

void NightControl::FileSave(wxCommandEvent& WXUNUSED(event))
{
  if( name.IsEmpty() ) {
    SetStatusText(_("Nothing to save."));
    return;
  }

  wxFileDialog select(this,_("Choose of FITS file name to save"),dir,file,
		      _("FITS files (*.fits)|*.fits|All files (*.*)|*"),
		      wxSAVE|wxOVERWRITE_PROMPT|wxCHANGE_DIR);
  if (select.ShowModal() == wxID_OK && ::wxCopyFile(name,select.GetPath())) {
  
    name.Empty();
    SetStatusText(_("Image saved to")+select.GetPath());
  }
  else
    SetStatusText(_("Image save failed."));
  
}

void NightControl::Exitus(void)
{
  if( ExpProc ) {
    SetStatusText(_("Shutting down exposure in progress."));
    ExpProc->Kill(ExpPid,wxSIGTERM);
    start->SetValue(false);
  } 
  frame->Destroy();  
}

void NightControl::FileExit(wxCommandEvent& WXUNUSED(event))
{
  Exitus();
}

void NightControl::WmExit(wxCloseEvent& WXUNUSED(event))
{
  Exitus();
}


void NightControl::SetPreferences(wxCommandEvent& WXUNUSED(event))
{

  wxString t = _("xnightview: Preferences");
  pref = new NightPrefer(this,t,wxDEFAULT_DIALOG_STYLE|wxRESIZE_BORDER);

  NightConfig *conf = frame->Config();
  NightImage *image = frame->Image();

  pref->set_chip(conf->Chip());
  pref->set_legend(conf->Legend());
  pref->set_xsize(conf->MaxWidth());
  pref->set_ysize(conf->MaxHeight());
  pref->set_zsize(conf->ZoomSize());
  pref->set_half(conf->Zoom());

  pref->set_bstep(conf->Backstep());
  pref->set_box(conf->BoxSize());

  pref->set_local(conf->Localhost());
  pref->set_host(conf->Remotehost());
  pref->set_auth(conf->AuthKey());
  pref->set_port(conf->Port());


  if( pref->ShowModal() == wxID_OK ) {

    conf->Set_Chip(pref->get_chip());
    conf->Set_Legend(pref->get_legend());
    conf->Set_MaxWidth(pref->get_xsize());
    conf->Set_MaxHeight(pref->get_ysize());
    conf->Set_ZoomSize(pref->get_zsize());
    conf->Set_Zoom(pref->get_half());
    winzoom->SetHalf(conf->Zoom());
    winprofi->SetHalf(conf->Zoom());
    conf->Set_MaxWidth(pref->get_xsize());
    conf->Set_MaxHeight(pref->get_ysize());
    conf->Set_Legend(pref->get_legend());
    conf->Set_Backstep(pref->get_bstep());
    conf->Set_BoxSize(pref->get_box());
    image->BoxSize(conf->BoxSize());
    image->BstepSize(conf->Backstep());
    conf->Set_Localhost(pref->get_local());
    conf->Set_Remotehost(pref->get_host());
    conf->Set_AuthKey(pref->get_auth());
    conf->Set_Port(pref->get_port());
    connection_setup(conf->Localhost(),conf->Remotehost(),conf->Port(),
		     conf->AuthKey());

  }
  pref->Destroy();
}

void NightControl::AboutCamera(wxCommandEvent& WXUNUSED(event))
{
  SetStatusText(_("Getting camera info..."));

  InfoProc = new wxProcess(this,ID18);
  InfoProc->Redirect();
  wxExecute(_("night_power info"),wxEXEC_ASYNC,InfoProc);
}

void NightControl::OnEndInfo(wxProcessEvent& event)
{

  if( event.GetExitCode() == 0 ) {
    SetStatusText(_("Camera info displayed."));

    size_t twidth = 0;
    size_t n = 0;
    wxString info;
    if( InfoProc && InfoProc->IsInputAvailable() ) {
      wxInputStream *is = InfoProc->GetInputStream();
      wxTextInputStream text(*is);
      while( ! is->Eof() ) {
	wxString t = text.ReadLine();
	if( t.Len() > twidth )
	  twidth = t.Len();
	info.Append(t+_("\n"));
	n++;
      }
    }
    delete InfoProc;

    wxDialog *d = new wxDialog(this,wxID_ANY,_("xnightview: camera info"),
			       wxDefaultPosition,wxDefaultSize,
			       wxCAPTION|wxCLOSE_BOX|wxRESIZE_BORDER);
    wxBoxSizer *t = new wxBoxSizer(wxVERTICAL);
    wxTextCtrl *tc = new wxTextCtrl(d, wxID_ANY, _(""), wxDefaultPosition,
				    wxDefaultSize,wxTE_MULTILINE|wxTE_READONLY);
    wxFont fonts = wxSystemSettings::GetFont(wxSYS_ANSI_FIXED_FONT);
    wxFont font(fonts.GetPointSize(),wxFONTFAMILY_MODERN,wxFONTSTYLE_NORMAL,wxFONTWEIGHT_NORMAL);
    wxTextAttr ta;
    ta.SetFont(font);
    tc->SetDefaultStyle(ta);
    tc->AppendText(info);
    t->Add(tc,wxSizerFlags(2).Expand().Border(wxALL, BWIDTH));
    t->Add(d->CreateButtonSizer(wxOK),wxSizerFlags().Right());

    // empirically set window size
    wxString tw(wxChar('*'),twidth+5);
    int text_width,text_height;
    GetTextExtent(tw,&text_width,&text_height,NULL,NULL,&font);
    text_height = (n+2)*text_height;
    d->SetSizerAndFit(t);
    d->SetSize(-1,-1,text_width,text_height);
    d->Show();
  }
  else
    SetStatusText(_("No info available."));
}

void NightControl::AboutFits(wxCommandEvent& WXUNUSED(event))
{
  NightImage *image = frame->Image();

  if( image->Presented() ) {
    wxDialog *d = new wxDialog(this,wxID_ANY,_("xnightview: FITS header"),
			       wxDefaultPosition,wxDefaultSize,
			       wxCAPTION|wxCLOSE_BOX|wxRESIZE_BORDER);
    wxBoxSizer *t = new wxBoxSizer(wxVERTICAL);
    wxTextCtrl *tc = new wxTextCtrl(d, wxID_ANY, _(""), wxDefaultPosition,
				    wxDefaultSize,wxTE_MULTILINE|wxTE_READONLY);
    wxFont fonts = wxSystemSettings::GetFont(wxSYS_ANSI_FIXED_FONT);
    wxFont font(fonts.GetPointSize(),wxFONTFAMILY_MODERN,wxFONTSTYLE_NORMAL,wxFONTWEIGHT_NORMAL);
    wxTextAttr ta;
    ta.SetFont(font);
    tc->SetDefaultStyle(ta);
    wxArrayString head = image->Header();
    wxString header;
    for(size_t i = 0; i < head.GetCount(); i++)
      header.Append(head.Item(i)+_("\n"));
    tc->AppendText(header);
    t->Add(tc,wxSizerFlags(2).Expand().Border(wxALL, BWIDTH));
    t->Add(d->CreateButtonSizer(wxOK),wxSizerFlags().Right());

    // empirically set window size
    size_t twidth = 85;
    size_t n = 25;
    wxString tw(wxChar('*'),twidth+1);
    int text_width,text_height;
    GetTextExtent(tw,&text_width,&text_height,NULL,NULL,&font);
    text_height = (n+1)*text_height;
    d->SetSizerAndFit(t);
    d->SetSize(-1,-1,text_width,text_height);
    d->Show();

  }
  else
    SetStatusText(_("No FITS image available."));
}

void NightControl::AboutHelp(wxCommandEvent& WXUNUSED(event))
{
  int xmax,ymax;
  ::wxDisplaySize(&xmax,&ymax);
  xmax = (3*xmax)/4;
  ymax = ymax/2;


  wxDialog *d = new wxDialog(this,wxID_ANY,_("xnightview: On-line help"),
			     wxDefaultPosition,wxDefaultSize,
			     wxCAPTION|wxCLOSE_BOX|wxRESIZE_BORDER);
  wxBoxSizer *t = new wxBoxSizer(wxVERTICAL);

  wxTextCtrl *tc = new wxTextCtrl(d, wxID_ANY, _(""), wxDefaultPosition,
				  wxDefaultSize,wxTE_MULTILINE|wxTE_READONLY|wxTE_DONTWRAP|wxTE_AUTO_URL);

  wxFont fonts = wxSystemSettings::GetFont(wxSYS_ANSI_FIXED_FONT);
  wxFont bold = wxFont(fonts.GetPointSize(),wxFONTFAMILY_DEFAULT,wxFONTSTYLE_NORMAL,wxFONTWEIGHT_BOLD);

  wxString tw(wxChar('X'),85);
  int text_width,text_height;
  GetTextExtent(tw,&text_width,&text_height,NULL,NULL,&fonts);
  text_height = 30*text_height;
  
  tc->AppendText(_("\n"));
  tc->SetDefaultStyle(wxTextAttr(wxNullColour,wxNullColour,bold,wxTEXT_ALIGNMENT_CENTRE));
  tc->AppendText(_("XNightview\n"));
  tc->SetDefaultStyle(wxTextAttr(wxNullColour));
  tc->AppendText(_("\n"));
  tc->AppendText(_("XNightview is an application to control of a CCD camera. It provides a connection\n"));
  tc->AppendText(_("to nightview daemon (Connect button), temperature setup (Regulate button and temperature)\n"));
  tc->AppendText(_("text entry, start/stop exposure and exposure setup parameters (Exposure line) and setup\n"));
  tc->AppendText(_("some documentation parameters (Object, Observer).\n"));
  tc->AppendText(_("\n"));
  tc->AppendText(_("XNightview also provides some simple analysis of current image as intensity and position\n"));
  tc->AppendText(_("information, simple computation of FWHM and S/N ratio and estimation of the background level.\n"));
  tc->AppendText(_("\n"));
  tc->AppendText(_("\n"));
  tc->SetDefaultStyle(wxTextAttr(wxNullColour,wxNullColour,bold));
  tc->AppendText(_("Background\n"));
  tc->SetDefaultStyle(wxTextAttr(wxNullColour));
  tc->AppendText(_("The background level is estimated by the median of every ten pixel (may be changed via\n"));
  tc->AppendText(_("preferences as the Background step parameter). A lower value means slower computation\n"));
  tc->AppendText(_("and more precise estimate. A higher value is fast and relative un-precise. Both depends on\n"));
  tc->AppendText(_("the particular image and a possible estimation error is not important.\n"));
  tc->AppendText(_("\n"));
  tc->SetDefaultStyle(wxTextAttr(wxNullColour,wxNullColour,bold));
  tc->AppendText(_("FWHM\n"));
  tc->SetDefaultStyle(wxTextAttr(wxNullColour));
  tc->AppendText(_("The full width at half of maximum (FWHM) is estimated as the second moment of intensity\n"));
  tc->AppendText(_("peak above background. The value is mean of FWHMs along vertical and horizontal axis.\n"));
  tc->AppendText(_("The size of the neighborhood of FWHM computation is determined by the Box size via\n"));
  tc->AppendText(_("preferences. Unfortunately the value of the FWHM depends on box size and computation is\n"));
  tc->AppendText(_("done in rectangular area\n"));
  tc->AppendText(_("\n"));
  tc->SetDefaultStyle(wxTextAttr(wxNullColour,wxNullColour,bold));
  tc->AppendText(_("S/N\n"));
  tc->SetDefaultStyle(wxTextAttr(wxNullColour));
  tc->AppendText(_("The signal to noise ratio S/N is simple ratio of the signal above background against to\n"));
  tc->AppendText(_("noise in computation box (in area determined as in FWHM). The value is affected by the\n"));
  tc->AppendText(_("all noise contributed in background, so the one is usually uncorrected on unbiased (raw)\n"));
  tc->AppendText(_("images. In that case, the S/N may be used as relative indicator of the image quality.\n"));
  tc->AppendText(_("\n"));
  tc->AppendText(_("\n"));
  tc->AppendText(_("More information on http://www.physics.muni.cz/mb/nightview/.\n"));
  tc->AppendText(_("\n"));
  t->Add(tc,wxSizerFlags(2).Expand().Border(wxALL, BWIDTH));
  t->Add(d->CreateButtonSizer(wxOK),wxSizerFlags().Right());
  d->SetSizerAndFit(t);
  d->SetSize(-1,-1,text_width,text_height);
  d->Show();
}


void NightControl::AboutAbout(wxCommandEvent& WXUNUSED(event))
{
  wxString cp(COPYLEFT,wxConvUTF8);
  wxString ver(NIGHTVERSION,wxConvUTF8);
  wxString info;

  info.Printf(_("XNightView %s\nNightView interactive control center for CCD camera\n%s"),
	      ver.c_str(),cp.c_str());

  wxMessageBox(info,_("About XNightView"));
  SetStatusText(_("XNightView"));
}

void NightControl::PowerButton(wxCommandEvent& event)
{
  if( power->GetValue() ) {

    SetStatusText(_("Setting camera on..."));

    if( wxExecute(_("night_power login"),wxEXEC_SYNC|wxEXEC_NODISABLE) == 0 )
      SetStatusText(_("Connecting server ... done"));
    else {
      SetStatusText(_("Connection to server ... failed"));
      power->SetValue(false);
      return;
    }
    
    SetStatusText(_("Getting filter list..."));

    wxArrayString output;
    filters.Clear();

    if( wxExecute(_("night_filter -list"),output,wxEXEC_SYNC|wxEXEC_NODISABLE) == 0 ) {
    
      if( output.GetCount() >= 1 ) {
	wxStringTokenizer l(output.Last(),_(" ,"));
	while ( l.HasMoreTokens() ) {
	  wxString filter = l.GetNextToken();
	  filter.Replace(_("'"),_(""));
	  filters.Add(filter);
	}
      }
    }

    wxString filter;
    if( filters.GetCount() > 0 ) {

      if( wxExecute(_("night_filter info"),output,wxEXEC_SYNC|wxEXEC_NODISABLE) == 0 ) {
	if( output.GetCount() >= 1 ) {
	  filter = output.Last();
	  filter.Replace(_("'"),_(""));
	}
      }

      controls->Show(3,false);
      controls->Remove(filt);
      filt = new wxRadioBox(this,Filt_But,_(" Filter "),wxDefaultPosition, 
			    wxDefaultSize,filters,0,wxRA_SPECIFY_COLS);
      filt->SetStringSelection(filter);
      controls->Insert(3,filt,wxSizerFlags(1).Expand().Border(wxALL, BWIDTH));
      controls->Show(filt,true);
      controls->Layout();

    }
    else {

      filters.Add(_(""));
      controls->Show(3,false);
      controls->Remove(filt);
      filt = new wxRadioBox(this,Filt_But,_(" Filter "),wxDefaultPosition, 
			    wxDefaultSize,filters,0,wxRA_SPECIFY_COLS);
      controls->Insert(3,filt,wxSizerFlags(1).Expand().Border(wxALL, BWIDTH));
      controls->Show(filt,true);
      controls->Layout();
      filt->Enable(false);
    }

    SetStatusText(_("Setting temperature..."));
    
    double p = 0.0, t = 666.0;

    if( wxExecute(_("night_temperature get -ta"),output,wxEXEC_SYNC|wxEXEC_NODISABLE) == 0 && 
	output.GetCount() >= 1) {
      output.Last().ToDouble(&t);
    }

    if( wxExecute(_("night_temperature get -r"),output,wxEXEC_SYNC|wxEXEC_NODISABLE) == 0 && 
	output.GetCount() >= 1) {
      output.Last().ToDouble(&p);
    }

    if( wxExecute(_("night_temperature get -ts"),output,wxEXEC_SYNC|wxEXEC_NODISABLE) == 0 && 
	output.GetCount() >= 1) {
      output.Last().ToDouble(&t);
    }

    if( p > 1.0 ) {
      wxString v;
      v.Printf(_("%.1lf"),t);
      temp->SetValue(v);
      regulate->SetValue(true);
      regulate->SetLabel(_(" Regulated "));
    }

    SetStatusText(_("Finishing gui setup..."));

    wxSizerFlags flags = wxSizerFlags().Expand();
    etime->Enable(true);
    etype->Enable(true);
    ebin->Enable(true);
    start->Enable(true);
    
    temp->Enable(true);
    regulate->Enable(true);
    power->SetLabel(_(" Disconnect "));

    SetStatusText(_("Starting up temperature info .."));

    LastTemperature = 666.0;
    ExpProc = 0;
    TempIter = 0;
    TempTimer.Start(1000);

    SetStatusText(_("Ready"));

  } else {

    SetStatusText(_("Shutdown of gui .."));

    TempTimer.Stop();

    SetStatusText(_("Setting camera off..."));

    if( ExpProc ) {
      SetStatusText(_("Shutting down exposure in progress."));
      ExpProc->Kill(ExpPid,wxSIGTERM);
      start->SetValue(false);
    } 

    filters.Clear();
    filters.Add(_("filter"));

    controls->Show(3,false);
    controls->Remove(filt);
    filt = new wxRadioBox(this,Filt_But,_(" Filter "),wxDefaultPosition, 
    		       wxDefaultSize,filters,0,wxRA_SPECIFY_COLS);
    controls->Insert(3,filt,wxSizerFlags(1).Expand().Border(wxALL, BWIDTH));
    controls->Show(filt,true);
    controls->Layout();

    temp->Enable(false);
    regulate->Enable(false);

    etime->Enable(false);
    etype->Enable(false);
    ebin->Enable(false);
    start->Enable(false);
    filt->Enable(false);
    power->SetLabel(_(" Connect "));
    regulate->SetLabel(_(" Regulate "));

    SetStatusText(_("Nightview"));
    SetStatusText(_(""),1);
  }
}


void NightControl::RegulateButton(wxCommandEvent& event)
{
  wxString temperature = temp->GetValue();
  double t;
  if( ! temperature.ToDouble(&t)) {
    regulate->SetValue(false);
    return;
  }

  if( regulate->GetValue() ) {

    regulate->Enable(false);
    temp->Enable(false);

    SetStatusText(_("Temperature regulation set on ..."));

    RegProc = new wxProcess(this,ID13);
    RegProc->Redirect();
    wxString x;
    x.Printf(_("night_temperature set -t %le"),t);
    wxExecute(x,wxEXEC_ASYNC,RegProc);
  }
  else {

    regulate->Enable(false);
    temp->Enable(false);

    SetStatusText(_("Temperature regulation set off ..."));

    RegProc = new wxProcess(this,ID13);
    RegProc->Redirect();
    wxExecute(_("night_temperature set -off"),wxEXEC_ASYNC,RegProc);
  }
}

void NightControl::TemperatureText(wxCommandEvent& event)
{
  wxString temperature = temp->GetValue();
  double t;
  if( ! temperature.ToDouble(&t))
    return;

  if( setpoint ) {

    regulate->Enable(false);
    temp->Enable(false);
    regulate->SetValue(true);

    SetStatusText(_("Temperature set-point change ..."));

    RegProc = new wxProcess(this,ID13);
    RegProc->Redirect();
    wxString x;
    x.Printf(_("night_temperature set -t %le"),t);
    wxExecute(x,wxEXEC_ASYNC,RegProc);
  }
}

void NightControl::TemperatureChange(wxCommandEvent& event)
{
  wxString temperature = temp->GetValue();
  double t;
  if( ! temperature.ToDouble(&t))
    return;

  regulate->Enable(true);
  regulate->SetValue(false);
  regulate->SetLabel(_(" Set-point "));
  setpoint = true;
}

void NightControl::OnEndRegulate(wxProcessEvent& event)
{
  if( event.GetExitCode() == 0 ) {
    SetStatusText(_("Temperature regulate on."));
    TempIter = 0;
    if( regulate->GetValue() )
      regulate->SetLabel(_(" Regulated "));
    else
      regulate->SetLabel(_(" Regulate "));
  }
  else {
    SetStatusText(_("Temperature regulate failed."));
    regulate->SetValue(false);
    regulate->SetLabel(_(" Regulate "));
  }

  regulate->Enable(true);
  temp->Enable(true);

  delete RegProc;
}

void NightControl::ExposureButton(wxCommandEvent& event)
{
  NightConfig *conf = frame->Config();

  wxString time = etime->GetValue();
  double t;
  if( ! time.ToDouble(&t))
    t = 0.0;

  int bin = ebin->GetCurrentSelection();
  if( bin == wxNOT_FOUND )
    bin = 1;
  else
    bin += 1;
  int type = etype->GetCurrentSelection();
  if( type == wxNOT_FOUND ) 
    type = 0;

  int shutter;
  shutter = type == 1;

  // remove previously created temporary file
  if( ! name.IsEmpty() ) {
    ::wxRemoveFile(name);
    name.Empty();
  }

  wxFileName filename;
  name = filename.CreateTempFileName(_("xnightview_"));

  if( start->GetValue() ) {

    SetStatusText(_("Exposure setup ..."));

    wxString observer = eobs->GetValue();
    wxString object = eobj->GetValue();
    
    wxString x = _("night_exposure");

    x += wxString::Format(_(" -t %le"),t);
    x += wxString::Format(_(" -c %c"),conf->Chip());
    x += wxString::Format(_(" -b %d"),bin);

    if( shutter )
      x += _(" -s off");
    else
      x += wxString(_(" -s on"));

    if( ! object.IsEmpty() )
      x += wxString(_(" -object \"")) + object + _("\"");

    if( ! observer.IsEmpty() )
      x += wxString(_(" -observer \"")) + observer + _("\"");
  
    if( ! name.IsEmpty() )
      x += wxString(_(" -o ")) + name;

    x += _(" -pn ");

    start->SetLabel(_(" Stop "));

    TempTimer.Stop();
    ExpTimer.Start(100);
    ExpStat = 0;

    wxRect rect;
    GetStatusBar()->GetFieldRect(1,rect);
    progress = new wxGauge(GetStatusBar(),wxID_ANY,1,rect.GetPosition(),rect.GetSize());

    ExpProc = new wxProcess(this,ID16);
    ExpProc->Redirect();
    ExpPid = wxExecute(x,wxEXEC_ASYNC,ExpProc);

  }
  else {
    if( ExpProc ) 
      ExpProc->Kill(ExpPid,wxSIGTERM);
  }

  // save current values
  conf->Set_Exptime(t);
  conf->Set_Exptype(type);
  conf->Set_Binning(bin);
  conf->Set_Observer(eobs->GetValue());

}

void NightControl::OnExposure(wxTimerEvent& event)
{
  int e = 0, d = 0;
  double t,tmax;

  if( ExpProc && ExpProc->IsInputAvailable() ) {
    wxInputStream *is = ExpProc->GetInputStream();
        wxTextInputStream text(*is);
    if( ! is->Eof() ) {
      wxString line = text.ReadLine();
      SetStatusText(line);
      line.Replace(_("/"),_(" "));
      line.Replace(_("%"),_(" "));
      e = line.Replace(_("Exposing"),_(" "));
      d = line.Replace(_("Downloading"),_(" "));
      wxStringTokenizer ta(line,_(" "));
      if( ta.HasMoreTokens() ) {
	wxString l = ta.GetNextToken();
	l.ToDouble(&t);
      }
      if( e && ta.HasMoreTokens() ) {
	wxString l = ta.GetNextToken();
	l.ToDouble(&tmax);
      }
      else
	tmax = 100.0;
    }
    if( progress ) {
      progress->SetRange(int(tmax));
      progress->SetValue(int(t));
    }
    if( ExpStat == 1 && d == 1 ) {
      ExpTimer.Stop();
      ExpTimer.Start(100);
    }
    if( e ) 
      ExpStat = 1;
    if( d )
      ExpStat = 2;
  }

}
 
void NightControl::OnEndExposure(wxProcessEvent& event)
{
  ExpTimer.Stop();
  if( event.GetExitCode() == 0 && frame->LoadImage(name) ) {

    SetStatusText(_("Exposure finished successfully."));

  }
  else
    SetStatusText(_("Exposure acquire failed."));

  delete progress;
  progress = 0;
  delete ExpProc;
  ExpProc = 0;
  ExpStat = 0;

  TempTimer.Start();
  TempIter = 17;
 
  start->SetValue(false);
  start->SetLabel(_(" Start "));

  if( etype->GetCurrentSelection() == 2 && event.GetExitCode() == 0 ) {
    ContinueTimer.Start(7,wxTIMER_ONE_SHOT);
  }
}


void NightControl::OnContinue(wxTimerEvent& event)
{
  start->SetValue(true);
  wxCommandEvent e;
  ExposureButton(e);
}

void NightControl::FilterSwitch(wxCommandEvent& event)
{

  wxString filter = filt->GetStringSelection();

  SetStatusText(_("Selecting filter..."));
  filt->Enable(false);
  start->Enable(false);

  FilterTimer.Start(200);
  FilterIter = 0;

  wxRect rect;
  GetStatusBar()->GetFieldRect(1,rect);
  fprogress = new wxGauge(GetStatusBar(),wxID_ANY,1,rect.GetPosition(),rect.GetSize());
  fprogress->SetRange(5);

  FiltProc = new wxProcess(this,ID14);
  FiltProc->Redirect();
  wxExecute(_("night_filter -f ")+filter,wxEXEC_ASYNC,FiltProc);

}

void NightControl::OnFilterSwitch(wxTimerEvent& event)
{
  FilterIter++;
  if( FilterIter > 5 )
    FilterIter = 0;
  if( fprogress )
    fprogress->SetValue(FilterIter);
}

void NightControl::OnEndFilterSwitch(wxProcessEvent& event)
{
  FilterTimer.Stop();
  if( event.GetExitCode() == 0 ) {
    SetStatusText(_("Filter selected."));
  }
  else {
    SetStatusText(_("Filter selection failed."));
  }
  delete fprogress;
  fprogress = 0;
  delete FiltProc;
  FiltProc = 0;
  
  filt->Enable(true);
  start->Enable(true);
}

void NightControl::UpdateFluxEtc()
{
  NightImage *image = frame->Image();

  long x = frame->XPointer();
  long y = frame->YPointer();

  int i = x + 1;
  int j = ((image->Height() - 1) - y) + 1;
  float f = image->Intensity(i,j);
  float fx = image->xFwhm(i,j);
  float fy = image->yFwhm(i,j);
  float sn = image->Sn(i,j);
  float m = image->Mean();

  wxString v;
  v.Printf(_("%d   %d"),i,j);
  coordinate->SetLabel(v);
  v.Printf(_("%.3f"),f);
  value->SetLabel(v);
  v.Printf(_("%.2f  %.1f"),(fx+fy)/2.0,sn);
  fwhm->SetLabel(v);
  v.Printf(_("%.1f"),m);
  mean->SetLabel(v);
}

void NightControl::UpdateGraphs()
{
  wxPaintEvent event;
  winzoom->OnPaint(event);
  winprofi->OnPaint(event);
}

void NightControl::OnTempUpdate(wxTimerEvent& event)
{

  TempIter++;

  if( TempIter > 125 ) {
    if( TempIter % 60 == 0 )
      TempUpdate();
    return;
  }

  if( TempIter > 60 ) {
    if( TempIter % 10 == 0 )
      TempUpdate();
    return;
  }

  if( TempIter > 30 ) {
    if( TempIter % 5 == 0 )
      TempUpdate();
    return;
  }

  if( TempIter > 10 ) {
    if( TempIter % 2 == 0 )
      TempUpdate();
    return;
  }

  if( TempIter <= 10 ) {
    TempUpdate();
    return;
  }
}

void NightControl::TempUpdate()
{
  if( ! TempProc ) {
    TempProc = new wxProcess(this,ID12);
    TempProc->Redirect();
    wxExecute(_("night_temperature get"),wxEXEC_ASYNC,TempProc);
  }
}

void NightControl::EndTempUpdate(wxProcessEvent& event)
{

  double t = 666.0, p = 0.0;

  if( TempProc && TempProc->IsInputAvailable() ) {
    wxInputStream *is = TempProc->GetInputStream();
    wxTextInputStream text(*is);
    while( ! is->Eof() ) {
      wxStringTokenizer ta(text.ReadLine(),_("="));

      wxString key,value;
      if( ta.HasMoreTokens() )
	key = ta.GetNextToken();
      if( ta.HasMoreTokens() )  
	value = ta.GetNextToken();

      if( key.StartsWith(_("ccd temperature")) )
	value.ToDouble(&t);

      if( key.StartsWith(_("cooling power")) )
	value.ToDouble(&p);
    }
    
    int q = int(rint(p));
    wxString line;
    line.Printf(_("%.1lf°C (%d%%)"),t,q);
    SetStatusText(line,1);
    LastTemperature = t;
  }
  delete TempProc;
  TempProc = 0;
}

void NightControl::connection_setup(bool localhost, wxString remotehost,
				    int port, wxString auth)
{
  if( localhost )
    wxUnsetEnv(_("NIGHTVIEW_HOST"));
  else {
    wxString tele_host;
    if( auth.IsEmpty() )
      tele_host.Printf(_("%s:%d"),remotehost.c_str(),port);
    else
      tele_host.Printf(_("%s@%s:%d"),
                       auth.c_str(),remotehost.c_str(),port);
    wxSetEnv(_("NIGHTVIEW_HOST"),tele_host.c_str());
  }
}
