

/*

  Xnightview

  $Id: fits.cpp,v 1.10 2008-02-12 23:04:24 hroch Exp $

*/

#include "xnightview.h"
#include <math.h>
#include <fitsio.h>

// For compilers that support precompilation, includes "wx/wx.h".
#include "wx/wxprec.h"

#ifdef __BORLANDC__
    #pragma hdrstop
#endif

// for all others, include the necessary headers (this file is usually all you
// need because it includes almost all "standard" wxWidgets headers)
#ifndef WX_PRECOMP
    #include "wx/wx.h"
#endif
#include <wx/tokenzr.h>


FitsImage::FitsImage(wxString name, int bskip, int bbox)
{
  fitsfile *f;
  long naxes[2];
  int status = 0;
  float nullval = 0;
  int fpixel = 1;
  int i;

  nx = 0;
  ny = 0;

  if( name.IsEmpty() ) {

    presented = false;
    return;

  }

  // open file
  fits_open_file(&f, name.fn_str(), READONLY, &status);
  fits_get_img_size(f,2,naxes,&status);

  // load header
  int nkey;
  char card[80];
  fits_get_hdrspace(f,&nkey,&i,&status);
  for(int n = 1; n <= nkey && status == 0; n++) {
    fits_read_record(f,n,card,&status);
    if( status != 0 ) break;
    wxString line(card,wxConvUTF8);
    head.Add(line);
  }

  // keywords
  for(int n = 1; n <= nkey && status == 0; n++ ) {
    char key[10];
    
    char v[80];
    char comment[80];
    fits_read_keyn(f,n,key,v,comment,&status);
    if( status != 0 ) break;
    
    wxString keyword(key,wxConvUTF8),value,val(v,wxConvUTF8);

    if( val.Find(_("'")) != wxNOT_FOUND ) {
      wxString v;     
      wxStringTokenizer tb(val,_("'"));
      for(int i = 0; i < 2 && tb.HasMoreTokens(); i++)
	v = tb.GetNextToken();
      if( v.Trim().IsEmpty() )
	value = _("");
      else
	value = v;
    }
    else
      value = val;

    if( keyword.StartsWith(_("OBJECT")) )
      object = value;
    if( keyword.StartsWith(_("DATE-OBS")) )
      dateobs = value;
    if( keyword.StartsWith(_("FILTER")) )
      filter = value;
    if( keyword.StartsWith(_("EXPTIME")) )
      exptime = value;
    if( keyword.StartsWith(_("XPIXSIZE")) )
      xpixsize = value;
    if( keyword.StartsWith(_("FOCUS")) )
      focus = value;
    if( keyword.StartsWith(_("XFACTOR")) )
      binning = value;

  }

  if( status == 0 ) {

    // image data
    long ndata = naxes[0]*naxes[1];
    data = new float[ndata];
    fits_read_img(f, TFLOAT, fpixel, ndata, &nullval, data, &i, &status);

    if( status != 0 )
      delete[] data;

  }

  fits_close_file(f, &status);

  if( status != 0 ) {

    presented = false;
    return;

  }
  else {

    presented = true;
    filename = name;
    nx = naxes[0];
    ny = naxes[1];
    bstep = bskip;
    box = bbox;
    Statistics();
  }
}

FitsImage::~FitsImage(void)
{
  if( presented )
    delete[] data;
}

void FitsImage::Statistics()
{
  if( presented ) {

    int nd = (nx*ny)/(bstep*bstep);
    if( nd == 0 ) nd = 1;
    float *d = new float[nd];
    
    for(int i = 0, j = 0; j < nd; i += bstep, j++)
      d[j] = data[i];
    sky = QMed(nd,d,nd/2+1);
 
    for(int i = 0, j = 0; j < nd; i += bstep, j++)
      d[j] = fabs(data[i] - sky);
    sig = QMed(nd,d,nd/2+1);

    mean = sky;
    dev = sig;

    delete[] d;

    if( fabs(dev) < 1.0/256.0 ) dev = 1.0;

    mean = mean - 3.0*dev;
    dev = dev/15.0;

  }
  else {
    mean = 0;
    dev = -1;
    sig = -1;
    sky = 0;
  }
}


float FitsImage::Intensity(int x, int y)
{
  if( presented && (0 <= x && x < nx) && (0 <= y && y < ny) )
    return *(data+y*nx + x);
  else
    return 0.0;
}

float FitsImage::xFwhm(int x, int y)
{
  if ( ! presented ) 
    return 0.0;

  double s = 0.0;
  double xf = 0.0;

  for(int i = -box; i < + box; i++) {
    double f = Intensity(i+x,y) - sky;
    if( f < 0.0 ) f = 0.0;
    s += f;
    xf += f*i*i;
  }
  if( s > 0.0 )
    return sqrt(xf/s);
  else
    return 0.0;
}

float FitsImage::yFwhm(int x, int y)
{
  if ( ! presented ) 
    return 0.0;

  double s = 0.0;
  double yf = 0.0;

  for(int j = -box; j < + box; j++) {
    double f = Intensity(x,j+y) - sky;
    if( f < 0.0 ) f = 0.0;
    s += f;
    yf += f*j*j;
  }
  if( s > 0.0 )
    return sqrt(yf/s);
  else
    return 0.0;
}

float FitsImage::Sn(int x, int y)
{
  if ( ! presented ) 
    return 0.0;

  double s = 0.0;
  int n = 0;

  for(int j = -box; j < + box; j++)
    for(int i = -box; i < +box; i++ ) {
      double f = Intensity(x+i,j+y) - sky;
      if( f < 0.0 ) f = 0.0;
      s += f;
      n++;
    }

  if( n > 0 && s > 0.0 )
    return s/sqrt(s + n*n*sig*sig);
  else
    return 0.0;
  
}

unsigned char FitsImage::EightBits(int x, int y)
{
  if( presented && (0 <= x && x < nx) && (0 <= y && y < ny) ) {
    int i8 = int(rint((*(data+y*nx+x) - mean)/dev)+0.5);

    if( i8 < 0 ) i8 = 0;
    if( i8 > 255 ) i8 = 255;

    return (unsigned char) i8;
  }
  else
    return 0;
}


float FitsImage::QMed(int n, float *a, int k)
{
  float w,x;
  int l,r,i,j;

  x = 0.0;

  l = 0;
  r = n - 1;

  while( l < r ) {
    x = a[k];
    i = l;
    j = r;

    do {

      while( a[i] < x )
        i++;
      while( x < a[j] )
        j--;

      if( i <= j ) {
        w = a[i];
        a[i] = a[j];
        a[j] = w;
        i++;
        j--;
      }
      
    } while ( i <= j );

    if( j < k ) l = i;
    if( k < i ) r = j;
  }

  return(x);
}
