
/*

  Xnightview

  $Id: xnightview.cpp,v 1.19 2008-03-30 00:41:20 hroch Exp $

*/


// For compilers that support precompilation, includes "wx/wx.h".
#include "wx/wxprec.h"

#ifdef __BORLANDC__
    #pragma hdrstop
#endif

// for all others, include the necessary headers (this file is usually all you
// need because it includes almost all "standard" wxWidgets headers)
#ifndef WX_PRECOMP
    #include "wx/wx.h"
#endif

// the application icon (under Windows and OS/2 it is in resources and even
// though we could still include the XPM here it would be unused)
#if !defined(__WXMSW__) && !defined(__WXPM__)
    #include "xnightview-icon.xpm"
#endif

#include <wx/propdlg.h>
#include <wx/wfstream.h>
#include <wx/txtstrm.h>
#include <wx/dcbuffer.h>
#include <wx/filename.h>
#include <wx/thread.h>

#include "xnightview.h"

IMPLEMENT_APP(XNightview)

DECLARE_EVENT_TYPE(wxEVT_MY_EVENT, -1)
  DEFINE_EVENT_TYPE(wxEVT_MY_EVENT)

BEGIN_EVENT_TABLE(NightFrame, wxFrame)
     EVT_SIZE(NightFrame::OnSize)
     EVT_CLOSE(NightFrame::OnClose)
     EVT_COMMAND(NightFrame_ID, wxEVT_MY_EVENT, NightFrame::SendName)
END_EVENT_TABLE()

BEGIN_EVENT_TABLE(NightZoom, wxWindow)
     EVT_PAINT(NightZoom::OnPaint)
END_EVENT_TABLE()

BEGIN_EVENT_TABLE(NightGraph, wxWindow)
     EVT_PAINT(NightGraph::OnPaint)
END_EVENT_TABLE()

BEGIN_EVENT_TABLE(NightDisplay, wxScrolledWindow)
     EVT_PAINT(NightDisplay::OnPaint)
     EVT_MOTION(NightDisplay::OnMouseRun)
     EVT_RIGHT_DOWN(NightDisplay::OnRightDown)
     EVT_KEY_DOWN(NightDisplay::OnKeyDown)
     EVT_SCROLLWIN(NightDisplay::OnScroll)
END_EVENT_TABLE()

bool XNightview::OnInit()
{
  wxString name;
  bool stdinp = false;

  for(int i = 1; i < argc; i++) {

    if( wxStrcmp(argv[i],_("-h")) == 0 || wxStrcmp(argv[i],_("--help")) == 0 ) {
      wxString cp(COPYLEFT,wxConvUTF8);
      wxString ver(NIGHTVERSION,wxConvUTF8);
      wxPrintf(_("XNightview %s, %s\nNightView interactive control center for camera.\n"),ver.c_str(),cp.c_str());
      wxPrintf(_("\n"));
      wxPrintf(_("Usage: xnightview [-] [-i fits]\n"));
      wxPrintf(_("options:\n"));
      wxPrintf(_("\t-i\tfits load FITS image\n"));
      wxPrintf(_("\t-\tget names of FITS images by reading of std. input\n"));
      wxPrintf(_("\t-h\tprint help\n"));
      wxPrintf(_("\n"));
      return(0);
    }

    else if( wxStrcmp(argv[i],_("-i")) == 0 && i++ <= argc )
      name = wxString(argv[i]);

    else if( wxStrcmp(argv[i],_("-")) == 0 )
      stdinp = true;

    else
      wxPrintf(_("An unknown command line paramater '%s' ignored.\n"),argv[i]);
  }

  NightFrame *frame = new NightFrame(name,stdinp,_("xnightview"));
  frame->Show(true);

  return true;
}

void NightDisplay::OnPaint(wxPaintEvent& WXUNUSED(event))
{

  wxPaintDC dc(this);
  DoPrepareDC(dc);

  NightFrame *frame = (NightFrame *) GetParent();
  NightImage *image = frame->Image();
  NightConfig *conf = frame->Config();

  wxBitmap *b = (wxBitmap *) image->xpm();

  wxMemoryDC mdc;
  mdc.DrawBitmap(*b,0,0,true);
  mdc.SelectObject(*b);

  if( conf->Legend() && image->Presented() ) {

    // legend

    wxString object = image->Object();
    wxString date = image->DateObs(); 
    wxString filter = image->Filter();
    wxString exp,e;
    e = image->ExpTime();
    if( ! e.IsEmpty() ) {
      double ee;
      e.ToDouble(&ee);
      if( ee > 10.0 ) 
	exp.Printf(_("%.0f sec"),ee);
      else
	exp.Printf(_("%.1f sec"),ee);
    }
    wxString text;
    if( ! object.IsEmpty() )
      text = _("  ")+image->Object();
    else
      text = _("  ?");
    if( ! filter.IsEmpty() )
      text.Append(_(",  ")+filter);
    if( ! date.IsEmpty() )
      text.Append(_(",  ")+date);
    if( ! exp.IsEmpty() )
      text.Append(_(",  ")+exp);

    wxColor color(_("white"));
    mdc.SetTextForeground(color);
    int text_width,text_height;
    GetTextExtent(text,&text_width,&text_height);
    int width,height;
    GetClientSize(&width,&height);
    width = image->Width();
    height = image->Height();
    int x0=0,y0=0;
    int x = 0;
    int y = height - text_height;
    mdc.DrawBitmap(image->Zoom(x0+x,y0+y,x0+x+width,y0+y+text_height,0.4),x,y,true);
    mdc.DrawText(text,x,y);

    // scale
    wxString f = image->Focus();
    wxString d = image->PixSize();
    wxString b = image->Binning();
    double ff,dd,bb,scale;
    f.ToDouble(&ff);
    d.ToDouble(&dd);
    b.ToDouble(&bb);
    if( ! f.IsEmpty() && ! d.IsEmpty() && ! b.IsEmpty() && fabs(ff) > 1e-3 ) {
      scale = ((1e-6*dd)/ff/bb)*57.3*3600.0*60.0;
      wxString v = _("1'");
      GetTextExtent(v,&text_width,&text_height);
      int slen = int(scale + 0.5);
      const int sep = 3;
      wxPen apen(color,1,wxSOLID);
      mdc.SetPen(apen);
      mdc.DrawLine(width-sep-slen,height - text_height,width-sep,height-text_height);
      mdc.DrawText(v,width-sep-slen/2-text_width/2,height-text_height);
    }
  }

  dc.Blit(0,0,b->GetWidth(),b->GetHeight(),&mdc,0,0);

}

// shift of scrolled image to origin
void NightDisplay::Origin()
{
  int width,height;
  GetClientSize(&width,&height);

  NightFrame *frame = (NightFrame *) GetParent();
  NightImage *image = frame->Image();
  
  if( image->Width() > width && image->Height() > height ){

    xoff = 0;
    yoff = image->Height()-height;
    Scroll(0,yoff);
    SetScrollPos(wxVERTICAL,yoff);

  }
  else if( image->Width() > width && image->Height() <= height ){


    xoff = 0;
    yoff = 0;

  }
  else if( image->Width() <= width && image->Height() > height ) {

    xoff = 0;
    yoff = image->Height()-height;
    Scroll(0,yoff);
    SetScrollPos(wxVERTICAL,yoff);

  }
  
}


void NightFrame::Resize()
{

  int w = image->Width();
  int h = image->Height();
  int wmax = conf->MaxWidth();
  int hmax = conf->MaxHeight();

  const int wthublen = 100;
  const int hthublen = 100;
    
  if( w <= wmax && h <= hmax ) {

    display->SetScrollRate(0,0);
    display->SetClientSize(w,h);
      
  }
  else if( w <= wmax && h > hmax ) {

    display->SetScrollRate(0,1);
    display->SetScrollbar(wxVERTICAL,0,hthublen,hthublen+h-hmax);
    display->SetClientSize(w,hmax);

  }
  else if( w > wmax && h <= hmax ) {

    display->SetScrollRate(1,0);
    display->SetScrollbar(wxHORIZONTAL,0,wthublen,wthublen+w-wmax);
    display->SetClientSize(wmax,h);
  }
  else {

    display->SetScrollRate(1,1);
    display->SetScrollbar(wxHORIZONTAL,0,wthublen,wthublen+w-wmax);
    display->SetScrollbar(wxVERTICAL,0,hthublen,hthublen+h-hmax);
    display->SetClientSize(wmax,hmax);
    
  }

  SetClientSize(display->GetSize());
}

void NightDisplay::OnKeyDown(wxKeyEvent& event)
{
  long keycode = event.GetKeyCode();
  long x = event.GetX();
  long y = event.GetY();

  // exit xnightview application on Alt-Q or Alt-q
  if( event.AltDown() && (keycode == 113 || keycode == 81) ) {
    NightFrame *frame = (NightFrame *) GetParent();
    frame->Destroy();
  }

  switch(keycode) {
  case 316: 
    x--; break;
  case 317: 
    y--; break;
  case 318:
    x++; break;
  case 319:
    y++; break;
  }
  WarpPointer(x,y);

  NightFrame *frame = (NightFrame *) GetParent();
  frame->set_coo(x,y);
}


void NightDisplay::OnScroll(wxScrollWinEvent& event)
{

  long orient = event.GetOrientation();
  long pos = event.GetPosition();
  if( orient == wxHORIZONTAL )
     xoff = pos;
  if( orient == wxVERTICAL )
     yoff = pos;

  Scroll( xoff, yoff );

}

void NightDisplay::OnMouseRun(wxMouseEvent& event)
{
  long x = event.GetX();
  long y = event.GetY();


  NightFrame *frame = (NightFrame *) GetParent();
  frame->set_coo(x+xoff,y+yoff);
  NightControl *panel = (NightControl *) frame->Panel();
  panel->UpdateFluxEtc();
  panel->UpdateGraphs();

}

void NightGraph::OnPaint(wxPaintEvent& event)
{
  // setup canvas
  wxBufferedPaintDC dc(this);
  if( ! dc.Ok() ) return;

  dc.Clear();

  wxColor color  = wxSystemSettings::GetColour(wxSYS_COLOUR_WINDOWTEXT);
  wxColor bcolor = wxSystemSettings::GetColour(wxSYS_COLOUR_BACKGROUND);

  // color and width
  wxPen apen(color,2,wxSOLID); // axes
  wxPen ticpen(color,1,wxSOLID); // tics
  wxPen bpen(bcolor,1,wxSOLID); // image

  wxCoord w, h;
  dc.GetSize(&w, &h);

  NightFrame *frame = night_control->GetParent();
  NightImage *image = frame->Image();

  long x = frame->XPointer();
  long y = frame->YPointer();

  int i =  x + 0;
  int j = ((image->Height() -1) - y) + 1;

  // inverse image
  int x1 = x - (half - 1);
  int y1 = y - (half + 1);
  int x2 = x + (half + 1);
  int y2 = y + (half - 1);

  wxBitmap b(image->iZoomP(x1,y1,x2,y2,bcolor.Red(),bcolor.Green(),bcolor.Blue(),0.5));
  wxImage img = b.ConvertToImage();
  wxBitmap bmp(img.Scale(w,h));
  dc.SetBackgroundMode(wxTRANSPARENT);
  dc.DrawBitmap(bmp,0,0,true);

  wxBrush brush(_("black"),wxTRANSPARENT);
  dc.SetBrush(brush);

  // rectangle around graph area
  dc.SetPen(apen);
  dc.DrawRectangle(1,1,w-1,h-1);


  const int nd = half;//10; // nd ~ 5 (zoomer)

  width = w;
  height = h;

  // tics
  int cwidth = width/(2*nd);
  int cheight = height/(2*nd);
  if( cwidth < 1 ) cwidth = 1;
  if( cheight < 1 ) cheight = 1;

  dc.SetPen(ticpen);

  // x ticks
  for(int ii = cwidth; ii < width; ii += cwidth ) {
    dc.DrawLine(ii,1,ii,4);
    dc.DrawLine(ii,height-5,ii,height);
  }

  /* y ticks */
  for(int ii = cheight; ii < height; ii += cheight ) {
    dc.DrawLine(1,ii,4,ii);
    dc.DrawLine(width-5,ii,width,ii);
  }

  wxPen pen1(color,1,wxSOLID);

  float mean = image->Mean();

  wxPoint p[2*nd+1];

  // x-profile
  /* get max of peak */
  float max = 0;
  for(int l = 0; l < 2*nd; l++) {
    
    float f = image->Intensity(i+(l-nd),j);
    if( f > max )
      max = f;
  }
  max = max - 0.9*mean;
  if( max > 0.0 )
    max = 0.9*height/max;
  else
    max = 1.0;

  for(int l = 0; l < 2*nd; l++ ) {
    float f = max*(image->Intensity(i+(l-nd),j) - mean);
    int x = cwidth*(l+0);
    int y = int(height - f) - cwidth;
    if( y <= 1 )
      y = 1;
    else if( y >= height )
      y = height;
    p[l] = wxPoint(x,y);
  }
  dc.DrawLines(2*nd,p);

  // y-profile

  /* get max of peak */
  max = 0;
  for(int l = 0; l < 2*nd; l++) {
    
    float f = image->Intensity(i,j+(l-nd));
    if( f > max )
      max = f;
  }
  max = max - 0.9*mean;
  if( max > 0.0 )
    max = 0.9*width/max;
  else
    max = 1.0;

  for(int l = 0; l < 2*nd; l++ ) {
    float f = max*(image->Intensity(i,j+(nd-l)) - mean);
    int x = int(width - f) - cheight;
    int y = cheight*(l+2);
    if( x <= 1 )
      x = 1;
    else if( x >= width )
      x = width;
    p[l] = wxPoint(x,y);
  }
  dc.DrawLines(2*nd,p);

  dc.SetPen(wxNullPen);
}

void NightZoom::OnPaint(wxPaintEvent& event)
{
  wxBufferedPaintDC dc(this);
  if( ! dc.Ok() ) return;
  dc.Clear();

  NightFrame *frame = night_control->GetParent();
  
  int i = frame->XPointer();
  int j = frame->YPointer();

  int x1 = i - (half - 1);
  int y1 = j - (half + 1);
  int x2 = i + (half + 1);
  int y2 = j + (half - 1);

  wxCoord w, h;
  dc.GetSize(&w, &h);

  wxBitmap b(frame->Image()->Zoom(x1,y1,x2,y2));
  wxImage img = b.ConvertToImage();
  wxBitmap bmp(img.Scale(w,h));
  dc.DrawBitmap(bmp,0,0,true);
}

void NightDisplay::OnRightDown(wxMouseEvent& event)
{
  NightFrame *frame = (NightFrame *) GetParent();
  NightControl *panel = (NightControl *) frame->Panel();
  panel->View();
}

NightDisplay::NightDisplay(NightFrame *w):
  wxScrolledWindow((wxWindow *)w,Scroll_win)
{
  xoff = 0;
  yoff = 0;
}

NightFrame::NightFrame(wxString name,bool stdinp,wxString title):
  wxFrame(NULL, NightFrame_ID, title)
{
  SetIcon(wxICON(xnightview_icon));
  SetThemeEnabled(true);

  conf = new NightConfig(_("xnightviewrc"));

  xcoo = 1;
  ycoo = 1;
  
  image = new NightImage(name);


  display = new NightDisplay(this);
  display->SetFocus();

  if( image->Presented() )
    LoadImage(name);
  else {
    
    display->SetVirtualSize(265,265);
    display->SetScrollRate(0,0);
    display->SetClientSize(265,265);
    SetClientSize(display->GetSize());
  }

  panel = new NightControl(this,title+_(": control panel"),image);

  if( ! name.IsEmpty() || stdinp )
    panel->DisablePower();

  if( stdinp ) {

    im = new InputName(this);
    if( im->Create() != wxTHREAD_NO_ERROR ) {
      wxPrintf(_("Can't create thread!"));
    }
    if ( im->Run() != wxTHREAD_NO_ERROR )	{
      wxPrintf(_("Can't start thread!"));
    }
  }
}


void NightFrame::OnClose(wxCloseEvent& event)
{
  Destroy();
}

bool NightFrame::LoadImage(wxString name)
{
  NightImage *i = new NightImage(name,conf->Backstep(),conf->BoxSize());

  if( i->Presented() ) {

    delete image;
    image = i;

    display->SetVirtualSize(image->Width(),image->Height());
    Resize();
    display->Origin();

    wxPaintEvent e;
    wxPostEvent(display,e);
 
    // set window title
    wxFileName fname = name;
    SetTitle(_("xnightview: ")+fname.GetName());
    
    return true;
  }
  else {
    delete i;
    return false;
  }
}

void NightFrame::NameSender(const wxString& name) { 

  wxCommandEvent event( wxEVT_MY_EVENT, GetId() );
  event.SetString(name);
  GetEventHandler()->ProcessEvent(event);
}

