package sbig;

public final class SBIGCamera
{
    protected static int[]  buffer;
    protected static native int command(int opcode, int[] data,
		 short[] image, Object other);
    protected static native String show_error(int m);


    public static final int  IMAGING_CCD = 0;
    public static final int  TRACKING_CCD = 1;
    /* C struct is 700 bytes, total */


    /* these are filled in on return... */
    int  linux_version;
    int	 nmbr_bad_columns;		/* bad columns in imaging CCD */
    int[]  bad_columns = new int[4];	// 32 bytes thus far
    int  imaging_abg_type;		/* 0 no ABG, 1 ABG present */
    byte[]  serial_number = new byte[10];	// 46 bytes thus far
    byte  pad1, pad2;
    int  firmware_version;				// 50 + 2 bytes
    byte[]  camera_name = new byte[64];			// 64 bytes
    SBIGCameraInfo[] camera_info = new SBIGCameraInfo[2];	// 584 bytes



    /**
     *  Initialize ST7/8 on the given parallel port
     *  -------------------------------------------
     *
     *  0x378 is the standard parallel port
     */
    public SBIGCamera(int port) {
	this(port, true);
    }

    public SBIGCamera(int port, boolean need_shared) {
	int  i, k, m;
	buffer = new int[1024];
	int[] data = buffer;

	if (need_shared) {
	    System.out.println("SBIGCamera: loading shared libraries--");
	    try {
		//System.load("./libsbigjni.so");
		System.loadLibrary("sbigjni");
	    } catch (Throwable e) {
		/* BUG ALERT: java-linux (as of 1.1.7v1a) IS BROKEN! */
		/* IF THE LOAD FAILS, JAVA_LINUX CRAPS OUT, INSTEAD OF
		 * PROCESSING THIS CODE... */
		System.out.println("Cannot load 'sbigjni.so'");
		e.printStackTrace();
		System.out.println(System.getProperties());
		System.exit(1);
	    }
	    System.out.println("SBIGCamera: --done shared libraries");
	}

	if ((i = command(port, data, null, null)) < 0) {
	    throw new RuntimeException("SBIGCamera: No camera found "
		+ "on parallel port 0x" + Integer.toHexString(port) +
		"\nError Reported is: " + show_error(i));
	}
	linux_version = data[0];
	nmbr_bad_columns = data[1];
	k = 2;
	for (i = 0; i < 4; ++i)
	    bad_columns[i] = data[k++];
	imaging_abg_type = data[k++];
	m = 0;
	for (i = 0; i < 10; ++i) {
	    serial_number[i] = (byte)(data[k]>>m);
	    if ((m += 8) == 32) {
		m = 0;
		++k;
	    }
	}
	/* ignore two bytes left in data[k] */
	++k;

	firmware_version = data[k++];
	m = 0;
	for (i = 0; i < 64; ++i) {
	    camera_name[i] = (byte)(data[k]>>m);
	    if ((m += 8) == 32) {
		m = 0;
		++k;
	    }
	}
	for (int camera = 0; camera < 2; ++camera) {
	    camera_info[camera] = new SBIGCameraInfo();
	    camera_info[camera].nmbr_readout_modes = data[k++];
	    for (int mode = 0; mode < 12; ++mode) {
		camera_info[camera].readout_mode[mode] = new SBIGReadoutMode();
		SBIGReadoutMode  mp = camera_info[camera].readout_mode[mode];
		mp.mode = data[k++];
		mp.width = data[k++];
		mp.height = data[k++];
		mp.gain = data[k++];
		mp.pixel_width = data[k++];
		mp.pixel_height = data[k++];
	    }
	}
	if (k != 173) {
	     throw new Error("SBigInit: did not consume "
	       + (4*173) + " bytes, but " + 4*k);
	}
    }


    /*
     *  Get camera status
     *  -----------------
     */
    /**
     *  Load an SBIGStatus object with current camera status
     *  Returns its argument.
     */
    public SBIGStatus  get_status(SBIGStatus st) {
	int  ret;
	int[] status_buffer = buffer;
	ret = command((int)'s', status_buffer, null, null);
	if (ret != 0)
	    throw new RuntimeException(show_error(ret));

	st.imaging_ccd_status = status_buffer[0];
	st.tracking_ccd_status = status_buffer[1];
	st.fan_on = status_buffer[2];
	st.shutter_state = status_buffer[3];
	st.led_state = status_buffer[4];
	st.shutter_edge = status_buffer[5];
	st.plus_x_relay = status_buffer[6];
	st.minus_x_relay = status_buffer[7];
	st.plus_y_relay = status_buffer[8];
	st.minus_y_relay = status_buffer[9];
	st.pulse_active = status_buffer[10];
	st.temperature_regulation = status_buffer[11];
	st.temperature_setpoint = status_buffer[12];
	st.cooling_power = status_buffer[13];
	st.air_temperature = status_buffer[14];
	st.ccd_temperature = status_buffer[15];
	return st;
    }

    /* NEED THIS in SBIG.h!!!!!!!!!!!!!!! */
    public static final int LEAVE_SHUTTER = 0;
    public static final int OPEN_SHUTTER = 1;
    public static final int CLOSE_SHUTTER = 2;
    public static final int INITIALIZE_SHUTTER = 3;

    public static final int LED_OFF = 0;
    public static final int LED_ON = 1;
    public static final int LED_BLINK_SLOW = 2;
    public static final int LED_BLINK_FAST = 3;

    public void set_control(int shutter, boolean fan, int led) {
	int[] in = buffer;
	in[0] = (fan? 1 : 0);
	in[1] = shutter;
	in[2] = led;
	int ret = command('c', in, null, null);
	if (ret != 0)
	    throw new RuntimeException(show_error(ret));
    }


    /* 
     *  Cooling control
     *  ---------------
     *
     * Note, it is advised to turn off regulation, but leave the fan on, for
     * a minute or so prior to shutting down the camera.
     */

    /**
     * set temperature-regulated cooling at given temp (in deg C)
     */
    public void set_cooling(float setting) {
	int[]  in = buffer;
	in[0] = TEMP_REGULATION_ON;
	in[1] = (int)(10.0*setting + 0.5);
	in[2] = 0;
	int ret = command('t', in, null, null);
	if (ret != 0)
	    throw new RuntimeException(show_error(ret));
    }
    /**
     * set direct-drive cooling at setting (0..255)
     */
    public void set_cooling(int setting) {
	int[]  in = buffer;
	in[0] = TEMP_REGULATION_DIRECT_DRIVE;
	in[1] = 0;
	in[2] = setting&0xFF;
	int ret = command('t', in, null, null);
	if (ret != 0)
	    throw new RuntimeException(show_error(ret));
    }
    /**
     *  return cooling off
     */
    public void set_cooling_off() {
	int[]  in = buffer;
	in[0] = TEMP_REGULATION_OFF;
	in[1] = 0;
	in[2] = 0;
	int ret = command('t', in, null, null);
	if (ret != 0)
	    throw new RuntimeException(show_error(ret));
    }
    private static final int  TEMP_REGULATION_OFF = 0;
    private static final int  TEMP_REGULATION_ON = 1;
    private static final int  TEMP_REGULATION_DIRECT_DRIVE = 2;



    /*
     *     Make an exposure
     *     ----------------
     */

    /*
     * Antiblooming gate control values
     */
    public final static int  ABG_OFF = 0;
    public final static int  ABG_LOW = 1;
    public final static int  ABG_MEDIUM = 2;
    public final static int  ABG_HIGH = 3;
    /**
     * Shutter control.  0 for no change in shutter (i.e. control of shutter
     * using SBigControl class), 1 normal shutter (open during exposure,
     * closed otherwise), 2 shutter closed (for taking dark frames).
     */
    public final static int  EXPOSE_SHUTTER_UNCHANGED = 0;
    public final static int  EXPOSE_SHUTTER_NORMAL = 1;
    public final static int  EXPOSE_SHUTTER_CLOSED = 2;

    /*
     *  Take a normal exposure on the given ccd for the given time
     *  (in seconds).  Use readout (below) to retrieve the exposure
     */
    public void expose(int ccd, float time) {
	expose(ccd, time, EXPOSE_SHUTTER_NORMAL, ABG_MEDIUM);
    }

    /*
     *  Take a dark frame on the given ccd for the given time
     *  (in seconds).  Use readout (below) to retrieve the exposure
     */
    public void expose_dark(int ccd, float time) {
	expose(ccd, time, EXPOSE_SHUTTER_CLOSED, ABG_MEDIUM);
    }

    /**
     * Take an exposure.
     * length of the exposure in hundreths of a second.  Allowable
     *    values are between 11 (0.11 sec) and 16777716 (167777.16 sec) for
     *    the imaging CCD and between 11 and 65535 (655.35 sec) for the
     *    tracking CCD.
     */
    public void expose(int ccd, float time, int shutter, int abg_state) {
 	int[]  in = buffer;
	int  exposure_time = (int)(100.0*time + 0.5);
	in[0] = ccd;
	in[1] = Math.max(11, Math.min(exposure_time, 65535));
	in[2] = abg_state;
	in[3] = shutter;
	int  ret = command('e', in, null, null);
	if (ret != 0)
	    throw new RuntimeException("SBIGCamera.expose: " + show_error(ret));
    }


    /*
     *   Read an image from the camera
     *   -----------------------------
     *
     *  Once you have an exposure, you need to read it from the camera.
     *  This is lengthy.
     *  Use one of the forms of readout_image() below.  You specify which
     *  ccd you are reading, the binning mode, and what part of the image
     *  you want to read.  You can also specify a callback method using
     *  an instance of SBIGReadoutCallback (see below) that will be called
     *  periodically during the readout.  This can be used to cancel the
     *  readout, update a progress bar, let the mouse respond, etc.
     */
    public static final int BIN_1X1 = 0;
    public static final int BIN_2X2 = 1;
    public static final int BIN_3X3 = 2;

    /**
      *  These two methods return the width and height respectively
      *  of an image on the given ccd with the giving binning.
      */
    public int get_image_width(int ccd, int binning) {
	try {
	    return camera_info[ccd].readout_mode[binning].width;
 	} catch (Exception e) {
	    return -1;
	}
    }
    public int get_image_height(int ccd, int binning) {
	try {
	    return camera_info[ccd].readout_mode[binning].height;
 	} catch (Exception e) {
	    return -1;
	}
    }

    /*
     * read the entirety of an image with no progress notification
     */
    public void readout_image(int ccd, int binning, short image_data[]) {
	readout_image(ccd, binning, 0, 0, 0, 0, image_data, null);
    }

    /*
     * read the entirety of an image with progress notification
     */
    public void  readout_image(int ccd, int binning, short image_data[],
				SBIGReadoutCallback notifier) {
	readout_image(ccd, binning, 0, 0, 0, 0, image_data, notifier);
    }

    public void  abort_image(int ccd) {
	int[] in = buffer;
	in[0] = ccd;
	in[1] = 0;
	in[2] = 0;
	in[3] = 0;
	in[4] = 0;
	in[5] = 0;
	in[6] = 0;  /* data buffer */
	in[7] = 0;
	in[8] = 0;  /* callback */
	int  ret = command('r', in, null, null);
	if (ret != 0) {
	    throw new RuntimeException("SBIGCamera.abort_image: " +
			 show_error(ret));
	}
    }

    /*
     * Read an image, with optional progress notification.
     *
     * If the width and/or height is zero, it is calculated.
     * image_data must be a non-null array of shorts whose size is
     * at least width*height.  The values are returned in it.  Each
     * has a range of 0..65535, so read a pixel value, it must be
     * converted to an int and masked with 0xFFFF.
     *
     * The progress notifier is a class that extends SBIGReadoutCallback
     * (see below), whose 'callback' method is invoked periodically
     * as the image downloads.  In general, the system freezes during
     * download because of the design of SBIG's download protocol.
     * However, the system will unfreeze during each progress notification
     * callback, so a progress bar can be updated, the readout can be
     * aborted, etc.
     */
    public void  readout_image(int ccd, int binning, int x, int y,
		  int width, int height, short[] image_data,
		  SBIGReadoutCallback notifier) {
	readout_image(ccd, binning, x, y, width, height, image_data,
	  notifier, false);
    }

    public void  readout_image(int ccd, int binning, int x, int y,
		  int width, int height, short[] image_data,
		  SBIGReadoutCallback notifier, boolean no_delay) {
        /* see class SBIGReadoutCallback below */
	if (width <= 0)
	    width = get_image_width(ccd, binning);
	if (height <= 0)
	    height = get_image_height(ccd, binning);
	if (image_data.length < width*height) {
	    throw new IllegalArgumentException("SBIGCamera.readout_image: " +
	      "image_data too short");
	}
	int[] in = buffer;
	in[0] = no_delay? 1 : 0;
	in[1] = ccd;
	in[2] = binning;
	in[3] = x;
	in[4] = y;
	in[5] = width;
	in[6] = height;
	in[7] = 0;  /* data buffer */
	in[8] = image_data.length;
	in[9] = 0;  /* callback */
	int  ret = command('r', in, image_data, notifier);
	if (ret != 0) {
	    throw new RuntimeException("SBIGCamera.readout_image: " +
			 show_error(ret));
	}
    }


    /*
     *  Filter wheel control
     *  --------------------
     *
     */

    /*
     * Set the CFW-8 filter position (1 to 5, inclusive)
     */
    public void set_cfw8(int position) {
	if (position < 1 || position > 5) {
	    throw new IllegalArgumentException("SBIGCamera.set_cfw8: " +
		"invalid position " + position);
	}
	pulse(60, 500 + 300*(position-1), 18270);
    }
    /*
     *  set pulses directly.
     *  (see SBIG's documentation for low-level filter wheel control
     *   on their web site)
     * nmbr_pulses;		-- 0 to 255 
     * pulse_width;		-- microsec, min 9
     * pulse_interval;		-- microsec, min 27+pulse_width
     */
    public void pulse(int nmbr_pulses, int pulse_width, int pulse_interval) {
	int[]  in = buffer;
	in[0] = nmbr_pulses;
	in[1] = pulse_width;
	in[2] = pulse_interval;
	int  ret = command('p', in, null, null);
    }

    /*
     * range is -1 : +1
     */
    public void deflect_ao7(double in_x, double in_y) {
	int x = (int)(2048.0*in_x);
	int y = (int)(2048.0*in_y);
	int[]  in = buffer;
	in[0] = Math.max(0, Math.min(x + 2048, 4095));
	in[1] = Math.max(0, Math.min(y + 2048, 4095));
	int  ret = command('d', in, null, null);
    }

    private static String _make_string(byte[] b) {
	int  i;
	for (i = 0; i < b.length; ++i)
	    if (b[i] == 0)
		break;
	return new String(b, 0, i) + "\n";
    }

    public String toString() {
	int  i;
	StringBuffer s = new StringBuffer("Camera: " +
				 _make_string(camera_name));
	s.append(" Linux_version: " + linux_version +
	  " firmware_version: " +  firmware_version + "\n");
	s.append(" Imaging ABG: ");
	s.append((imaging_abg_type != 0)? "Present\n" : "None\n");
	s.append(" Nmbr of bad columns in Imaging CCD: "
				 + nmbr_bad_columns + "\n");
	for (i = 0; i < nmbr_bad_columns; ++i)
	    s.append("  column " + bad_columns[i] + "\n");
	s.append(" Serial Number: " + _make_string(serial_number));
	SBIGReadoutMode  rmp;
	for (i = 0; i < camera_info[IMAGING_CCD].nmbr_readout_modes; ++i) {
	    rmp = camera_info[IMAGING_CCD].readout_mode[i];
	    s.append("ImagingCCD " + rmp.toString() + "\n");
	}
	for (i = 0; i < camera_info[TRACKING_CCD].nmbr_readout_modes; ++i) {
	    rmp = camera_info[TRACKING_CCD].readout_mode[i];
	    s.append("TrackingCCD " + rmp.toString() + "\n");
	}
	return s.toString();
    }
}
