/*   
      KONVE     Convert utility ST-xx  -> FITS   
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "konve.h"

#define MAXLINE    256     /* length of text line */
#define MAXFORMAT   10

#define KNAME "kout????.fits"
 
#define VERSION "KONVE Version 1.7, Copyright (C) 2001 F.Hroch, Masaryk University, Brno, CZ"

#define BIAS 100         /* SBIG cameras adds the bias to any pixel */
#define BZERO 32768      /* fits data = image * BSCALE + BZERO */
#define BSCALE 1  

char CType[MAXTYPE][30] =
 { "ST-4X Image",                          /* 0 */
   "ST-4X Compressed Image",               /* 1 */
   "ST-5 Image",                           /* 2 */
   "ST-5 Compressed Image",                /* 3 */
   "ST-6 Image",                           /* 4 */
   "ST-6 Compressed Image",                /* 5 */
   "ST-7 Image",                           /* 6 */
   "ST-7 Compressed Image",                /* 7 */
   "ST-8 Image",                           /* 8 */
   "ST-8 Compressed Image",                /* 9 */
/*
 *  added two file types by Miroslav Broz,
 *  miroslav.broz@usa.net, Nov 24th 1998
 */
   "PixCel255 Image",                      /* 10 */
   "PixCel255 Compressed Image" };         /* 11 */


int main ( argc, argv )
int argc;
char *argv[];
{
int konv(),pos(),copy(),endpos(),i,n,icit,icit0,stdinput = 0, head = 0;
int quiet = 0;
long tcor;
char dirname[MAXLINE] = "", parname[MAXLINE] = "",*j;
char line[MAXLINE],buf1[MAXLINE],mask[MAXLINE] = KNAME;
char flip[MAXLINE] = "";
FILE *fopen(), *fp;
static char format[MAXFORMAT][5] = 
 { "","%01d", "%02d", "%03d", "%04d", "%05d", "%06d", "%07d", "%08d", "%09d" };

icit = 1; icit0 = 1;
tcor = 0L;

for( i = 0; i < argc; i++ ) {              /* loop over all parameters */

  if( (j = strstr(argv[i],"@")) != NULL ) {                  /* dir file */
    if( strlen(argv[i]) == 1 )
      stdinput = 1;
    else
      strncpy(dirname,j + strlen("@"), MAXLINE);
  }
  if( (j = strstr(argv[i],"par=")) != NULL )          /* parameter file */
    strncpy(parname,j + strlen("par="),MAXLINE);

  if( strcmp(argv[i],"-p") == 0 )                  
    strncpy(parname,argv[++i],MAXLINE);
  
  if( (j = strstr(argv[i],"tcor=")) != NULL )        /* time correction */
    sscanf(j + strlen("tcor="),"%ld",&tcor);

  if( (j = strstr(argv[i],"mask=")) != NULL )   /* mask of output files */
    strncpy(mask,j + strlen("mask="),MAXLINE);

  if( (j = strstr(argv[i],"-o")) != NULL )  
    strncpy(mask,argv[++i],MAXLINE);

  if( strcmp(argv[i],"-flip") == 0 )
    strncpy(flip,argv[++i],MAXLINE);

  if( strcmp(argv[i],"-c") == 0 )
    sscanf(argv[++i],"%d",&icit0);

  if( strcmp(argv[i],"-h") == 0 )                      /* only header */
    head = 1;

  if( strcmp(argv[i],"-q") == 0 )                      /* quiet */
    quiet = 1;

  if( strcmp(argv[i],"-g") == 0 )
    if( (fp = fopen("konve.par","a")) != NULL ) {
      fprintf(fp,"OBJECT  = 'Star'\n");
      fprintf(fp,"OBSERVER= 'J.Walker'\n");
      fprintf(fp,"TELESCOP= '1.0m'\n");
      fprintf(fp,"RA      =            90.0\n");
      fprintf(fp,"DEC     =            90.0\n");
      fprintf(fp,"EQUINOX =          2000.0\n");
      fclose(fp);
      printf("'konve.par' is an example of the parameter file.\n");
      return (0);
    }

  if( strcmp(argv[i],"-L") == 0 ) {            /* license */
    printf("%s\n%s\n%s\n%s\n\n%s\n%s\n%s\n%s\n\n%s\n%s\n%s\n\n",
"    This program is free software; you can redistribute it and/or modify",
"    it under the terms of the GNU General Public License as published by",
"    the Free Software Foundation; either version 2, or (at your option)",
"    any later version.",
"    This program is distributed in the hope that it will be useful",
"    but WITHOUT ANY WARRANTY; without even the implied warranty of",
"    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the",
"    GNU General Public License for more details.",
"    You should have received a copy of the GNU General Public License",
"    along with this program; if not, write to the Free Software",
"    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.");
    return (0);
  }

  if( argc == 1 || strcmp(argv[i],"-help") == 0 ) {
    printf("%s\n\n",VERSION);
    printf("KONVE        Convert utility ST-xx  ->  FITS\n\n");
    printf("Usage:konve [options] [@dirfile] [par=parfile] [tcor=?] [mask=?] [image1] ...\n\n");
    printf("@dirfile= file with name(s) of image(s) to convert, no wildcards *,?\n");
    printf("          (for only '@', read image(s) name(s) from standart input)\n");
    printf("image1...= name(s) of simple image(s) to convert, no wildcards *,?\n");
    printf("par= parameter file\n");
    printf("tcor= correction of time in seconds\n");
    printf("mask= output mask (default = kout????.fts)\n");
    printf("options:\n");
    printf("\t-p <parname>\t parameter file (equivalent with 'par=')\n");
    printf("\t-o <mask>\t output mask (equivalent with 'mask=')\n");
    printf("\t-c <number>\t counter initial value (default = 1)\n");
    printf("\t-flip <x>,<y>,<xy>\t flipping around axis\n");
    printf("\t-q\t quiet\n");
    printf("\t-h\t create only header of FITS file\n");
    printf("\t-help\t give this help\n");
    printf("\t-L\t display software license\n");
    printf("Give '-g' switch for an example of the parameter file.\n");
    return (0);
  }
}

if( ! quiet ) printf("%s\n\n",VERSION);

if( (j = strstr(mask,"*")) != NULL )                /* remove '*' wildcard */
  *j = '?';
if( (j = strstr(mask,"?")) != NULL )                      /* number of '?' */
  for( n = 0; *j == '?' && n < MAXFORMAT-1; j++ )
     n = n + 1;
j = strstr(mask,"?");

icit = icit0;
                                  /* images names read from command line */
for( i = 1; i < argc; i++)
  if( strstr(argv[i],"=") == NULL && strstr(argv[i],"@") == NULL  &&  
    index(argv[i],'-') != argv[i] && index(argv[i-1],'-') != argv[i-1] ) {
    if( j != NULL ) {
      sprintf(buf1,format[n],icit);
      memcpy(j,buf1,n);
    }
    if( konv(argv[i],parname,tcor,mask,head,flip) < 0 )
      printf("Can't convert file: %s\n",argv[i]);
    icit = icit + 1; 
  }

if( strlen(dirname) > 0 ) {                   /* images names read from file */
  if( (fp = fopen(dirname,"r")) == NULL ) {
    printf("Can't open file %s\n",dirname);
    exit(0); 
  }
  while( fgets(buf1,MAXLINE,fp) != NULL ) {
    sscanf(buf1,"%s",line);
    if( j != NULL ) {
      sprintf(buf1,format[n],icit);
      memcpy(j,buf1,n); 
    }
    if( konv(line,parname,tcor,mask,head,flip) < 0 )
      printf("Can't convert file %s \n",line);
    icit = icit + 1;
  }
  fclose(fp);
}
                                      /* images names from standart input */
if( stdinput )
  while( fgets(buf1,MAXLINE,stdin) != NULL ) {
    sscanf(buf1,"%s",line);
    if( j != NULL ) {
      sprintf(buf1,format[n],icit);
      memcpy(j,buf1,n);
    }
    if( konv(line,parname,tcor,mask,head,flip) < 0 )
      printf("Can't convert file: %s\n",line);
    icit = icit + 1; 
  }

if( argc > 1  && icit == icit0 )
  printf("No input files.\n");

return 0;
}


/************************* Conversion ********************************/

int konv(name,parname,tcor,ftsname,head,flip)
  char name[],parname[],ftsname[];
  long tcor;
  int head;
  char flip[];
{
  FILE *STopen(),*FITScreate(),*fp,*fs;
  int i, l, n, ftype, nrow, ncol, value, hi, lo;
  int getSTrow(), putFITSrow(), FITSclose();
  unsigned char *stbuf,temp;
  signed char *image;
  int endpos(),identify();


  image = NULL;
  stbuf = NULL;

printf("%s -> %s\n",name,ftsname);
  
if( (fp = STopen(name,"rb")) == NULL ) {
    printf("Can't open the input file.\n");
    return(-1);
 }

if( (stbuf = malloc(HEADSIZE)) == NULL) {  /* header of ST-xx file */
   printf("Can't create internal buffer.\n");
   fclose(fp);
   return(-1);
 }
  
if( (n = fread(stbuf,1,HEADSIZE,fp)) != HEADSIZE ) {
    printf("%d %d",n,HEADSIZE);
    printf("Can't read the head from the input file.\n");
    return(-1);
 }

if( (ftype = identify(stbuf)) < 0 ) {  /* identification of file type */
    fclose(fp);
    return(-1);
 }

nrow = atol(stbuf+endpos(stbuf,"Width ="));  /* size */
ncol = atol(stbuf+endpos(stbuf,"Height ="));
if( nrow < 1 || ncol < 1 ) {
    printf("Error in file format (width or height).\n");
    fclose(fp);
    return(-1);
 }

/* create FITS file */
if( (fs = FITScreate(ftsname,"wb",stbuf,parname,tcor)) == NULL ) {
    printf("Can't create FITS file.\n");
    fclose(fp);
    return(-1);
 }
 free(stbuf);

 if( head ) goto konec;  /* do not convert image data */

 if( (stbuf = malloc( 2*nrow )) == NULL ) { 
   printf("Can't create internal buffer.\n");
   fclose(fp);
   return(-1);
 }
  
 if( (image = malloc( 2*nrow*ncol )) == NULL ){
   printf("Can't create image buffer.\n");
   fclose(fp);
   return(-1);
 }

 /* read image data */
 l = 0; 
 while( (n = getSTrow(fp,stbuf,ftype,nrow)) > 0 ) {
   if( n == -1 || l >= 2*ncol*(nrow - 1) ) {
     printf("Can't read data from the ST file.\n");
     goto konec; 
   }
 
   if( strstr(flip,"y") != NULL || strstr(flip,"Y") != NULL ) /* flipping */
     for( i = 0; i < nrow/2; i++ ) {   /* flipping */ 
       /*         a[2*i]   <-> a[2*nrow - 2*i - 2]
		  a[2*i+1] <-> a[2*nrow - 2*i - 1] */
       temp = *(stbuf + 2*i);
       *(stbuf + 2*i) = *(stbuf + 2*(nrow - i) - 2);
       *(stbuf + 2*(nrow - i) - 2) = temp;
       temp = *(stbuf + 2*i + 1);
       *(stbuf + 2*i + 1) = *(stbuf + 2*(nrow - i) - 1);
       *(stbuf + 2*(nrow - i) - 1) = temp;        
     }

   for( i = 0; i < 2*nrow; i = i + 2 ) {
     /*
     value = *(stbuf + i) + 256* *(stbuf + i + 1);
     value = value - BIAS;             
     if( value < 0 ) value = 0;    
     if( value > 65535 ) value = 65535;
     value = (value - BZERO)/BSCALE;  
     hi = value / 256;
     lo = value % 256;
     if( hi < 0 && lo != 0 ) {
       hi = hi - 1;
     }
     */
     lo = *(stbuf + i);
     hi = *(stbuf + i + 1) - 128; 
     *(image + l) = hi;
     *(image + l + 1) = lo;
     /*printf("%d: %u %u -> %d %d\n",value,*(stbuf+i),*(stbuf+i+1),hi,lo);*/
     l = l + 2;
   }
}

if( strstr(flip,"x") != NULL || strstr(flip,"X") != NULL ) /* flipping */
  for( i = 0; i < ncol; i = i + 1 ) {
    if( putFITSrow(fs,image + 2*i*nrow,nrow) != nrow ) {
      printf("Can't write image data to the FITS file.\n");
      goto konec;
    }
  }
else
  for( i = ncol - 1; i >= 0; i = i - 1 ) { 
    if( putFITSrow(fs,image + 2*i*nrow,nrow) != nrow ) {
      printf("Can't write image data to the FITS file.\n");
      goto konec;
    }
  }

konec:

/* close files and return */
  if( image != NULL) free(image);
  if( stbuf != NULL) free(stbuf);
  fclose(fp);
  if( FITSclose(fs) != FITSSIZE ) {
    printf("Can't close the FITS file.\n");
    return(-1);
  }
  return(0);
}


/************* identification of the image type ************************/

/* This function return a '0' for uncompressed image type
                   and an '1' for compressed type.
*/
int identify( s )
  unsigned char *s;
{
  int i,pos();

  for( i = 0; i < MAXTYPE; i++ )
    if( pos(s,CType[i]) >= 0 )
      return(i - (i/2)*2);

  printf("Unknown image type.\n");
  return(-1);
}

